/* Copyright 2023 Yann ORLAREY
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <algorithm>
#include <cmath>
#include <functional>
#include <random>

#include "check.hh"
#include "interval_algebra.hh"
#include "interval_def.hh"

namespace itv {
//------------------------------------------------------------------------------------------
// Interval Sin
// interval Sin(const interval& x);
// void testSin();

static double sinPi(double x)
{
    return std::sin(x * M_PI);
}

interval interval_algebra::Sin(const interval& x)
{
    if (x.isEmpty()) {
        return empty();
    }

    int precision = exactPrecisionUnary(sin, 0.5, std::pow(2, x.lsb()));
    if ((precision == INT_MIN) || taylor_lsb) {
        precision =
            2 * x.lsb() - 1;  // if x.lsb() is so small that the automatic computation doesn't work
    }

    if (x.size() >= 2 * M_PI) {
        return {-1, 1, precision};
    }

    // normalize input interval between 0..2
    double l = std::fmod(x.lo(), 2 * M_PI);
    if (l < 0) {
        l += 2 * M_PI;
    }
    interval i(l, l + x.size(), x.lsb());

    // compute the default boundaries
    double a  = std::sin(i.lo());
    double b  = std::sin(i.hi());
    double lo = std::min(a, b);
    double hi = std::max(a, b);

    // check if integers are included
    if (i.has(M_PI_2) || i.has(5 * M_PI_2)) {
        hi = 1;
    }
    if (i.has(3 * M_PI_2) || i.has(7 * M_PI_2)) {
        lo = -1;
    }

    double v = M_PI_2;  // value of the interval at which the finest precision is computed
                        // defaults at 0.5, interchangeable with any other half-integer

    // precision if we don't hit the half integers
    if (i.hi() < M_PI_2) {
        v = x.hi();
    } else if (((i.lo() > M_PI_2) && (i.hi() < 3 * M_PI_2)) ||
               ((i.lo() > 3 * M_PI_2) && (i.hi() < 2.5 * M_PI))) {
        double delta_hi = std::ceil(i.hi() / M_PI + 0.5) - i.hi() / M_PI;
        double delta_lo = i.lo() / M_PI - std::floor(i.lo() / M_PI - 0.5);
        if (delta_lo > delta_hi) {  // if i.hi is closer to its higher half-integer than i.lo() to
                                    // its lower half-integer
            v = x.hi();
        } else {
            v = x.lo();
        }
    }

    precision = exactPrecisionUnary(std::sin, v, std::pow(2, x.lsb()));
    if ((precision == INT_MIN) || taylor_lsb) {
        if (v != 0.5 * M_PI) {
            precision = x.lsb() + (int)std::floor(std::log2(std::abs(std::cos(
                                      v))));  // (int)floor(log2(M_PI*cos(M_PI*v))) + x.lsb();
        } else {
            precision = 2 * x.lsb() - 1;  // - (int)floor(2*log2(M_PI));
        }
    }

    return {lo, hi, precision};
}

void interval_algebra::testSin()
{
    // analyzeUnaryMethod(5, 20000, "sin", interval(-1, 1, -3), std::sin, &interval_algebra::Sin);
    analyzeUnaryMethod(10, 40000, "sin", interval(0, 2 * M_PI, -3), std::sin,
                       &interval_algebra::Sin);
    analyzeUnaryMethod(10, 40000, "sin", interval(0, 2 * M_PI, -5), std::sin,
                       &interval_algebra::Sin);
    analyzeUnaryMethod(10, 40000, "sin", interval(0, 2 * M_PI, -10), std::sin,
                       &interval_algebra::Sin);
    analyzeUnaryMethod(10, 40000, "sin", interval(0, 2 * M_PI, -15), std::sin,
                       &interval_algebra::Sin);
    analyzeUnaryMethod(10, 40000, "sin", interval(0, 2 * M_PI, -20), std::sin,
                       &interval_algebra::Sin);
    analyzeUnaryMethod(10, 40000, "sin", interval(0, 2 * M_PI, -24), std::sin,
                       &interval_algebra::Sin);
}
}  // namespace itv
