// Copyright 2023 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package nss

import (
	"crypto/x509"
	"encoding/pem"
	"reflect"
	"sort"
	"strings"
	"testing"
	"time"
)

func mustParse(b []byte) *x509.Certificate {
	block, _ := pem.Decode(b)
	cert, err := x509.ParseCertificate(block.Bytes)
	if err != nil {
		panic(err)
	}
	return cert
}

var testComodo = mustParse([]byte(`-----BEGIN CERTIFICATE-----
MIIEMjCCAxqgAwIBAgIBATANBgkqhkiG9w0BAQUFADB7MQswCQYDVQQGEwJHQjEb
MBkGA1UECAwSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHDAdTYWxmb3JkMRow
GAYDVQQKDBFDb21vZG8gQ0EgTGltaXRlZDEhMB8GA1UEAwwYQUFBIENlcnRpZmlj
YXRlIFNlcnZpY2VzMB4XDTA0MDEwMTAwMDAwMFoXDTI4MTIzMTIzNTk1OVowezEL
MAkGA1UEBhMCR0IxGzAZBgNVBAgMEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UE
BwwHU2FsZm9yZDEaMBgGA1UECgwRQ29tb2RvIENBIExpbWl0ZWQxITAfBgNVBAMM
GEFBQSBDZXJ0aWZpY2F0ZSBTZXJ2aWNlczCCASIwDQYJKoZIhvcNAQEBBQADggEP
ADCCAQoCggEBAL5AnfRu4ep2hxxNRUSOvkbIgwadwSr+GB+O5AL686tdUIoWMQua
BtDFcCLNSS1UY8y2bmhGC1Pqy0wkwLxyTurxFa70VJoSCsN6sjNg4tqJVfMiWPPe
3M/vg4aijJRPn2jymJBGhCfHdr/jzDUsi14HZGWCwEiwqJH5YZ92IFCokcdmtet4
YgNW8IoaE+oxox6gmf049vYnMlhvB/VruPsUK6+3qszWY19zjNoFmag4qMsXeDZR
rOme9Hg6jc8P2ULimAyrL58OAd7vn5lJ8S3frHRNG5i1R8XlKdH5kBjHYpy+g8cm
ez6KJcfA3Z3mNWgQIJ2P2N7Sw4ScDV7oL8kCAwEAAaOBwDCBvTAdBgNVHQ4EFgQU
oBEKIz6W8Qfs4q8p74Klf9AwpLQwDgYDVR0PAQH/BAQDAgEGMA8GA1UdEwEB/wQF
MAMBAf8wewYDVR0fBHQwcjA4oDagNIYyaHR0cDovL2NybC5jb21vZG9jYS5jb20v
QUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNqA0oDKGMGh0dHA6Ly9jcmwuY29t
b2RvLm5ldC9BQUFDZXJ0aWZpY2F0ZVNlcnZpY2VzLmNybDANBgkqhkiG9w0BAQUF
AAOCAQEACFb8AvCb6P+k+tZ7xkSAzk/ExfYAWMymtrwUSWgEdujm7l3sAg9g1o1Q
GE8mTgHj5rCl7r+8dFRBv/38ErjHT1r0iWAFf2C3BUrz9vHCv8S5dIa2LX1rzNLz
Rt0vxuBqw8M0Ayx9lt1awg6nCpnBBYurDC/zXDrPbDdVCYfeU0BsWO/8tqtlbgT2
G9w84FoVxp7Z8VlIMCFlA2zs6SFz7JsDoeA3raAVGI/6ugLOpyypEBMs1OUIJqsi
l2D4kF501KKaU73yqWjgom7C12yxow+ev+to51byrvLjKzg6CYG1a4XXvi3tPxq3
smPi9WIsgtRqAEFQ8TmDn5XpNpaYbg==
-----END CERTIFICATE-----`))

var testTrustcor = mustParse([]byte(`-----BEGIN CERTIFICATE-----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-----END CERTIFICATE-----`))

func TestParseCertData(t *testing.T) {
	trustcorDistrust, err := time.Parse("060102150405Z0700", "221130000000Z")
	if err != nil {
		t.Fatalf("failed to parse distrust time: %s", err)
	}

	for _, tc := range []struct {
		name   string
		data   string
		output []*Certificate
		err    string
	}{
		{
			name: "valid certs",
			data: validCertdata,
			output: []*Certificate{
				&Certificate{X509: testComodo},
				&Certificate{X509: testTrustcor, Constraints: []Constraint{DistrustAfter(trustcorDistrust)}},
			},
		},
		{
			name: "cert obj, no trust obj",
			data: certNoTrust,
			err:  "missing trust object for certificate with SHA1 hash: d1eb23a46d17d68fd92564c2f1f1601764d8e349",
		},
		{
			name: "trust obj, no cert obj",
			data: trustNoCert,
		},
		{
			name: "missing certificate der",
			data: missingCertificateOctal,
			err:  "malformed CKO_CERTIFICATE object",
		},
		{
			name: "untrusted cert (cert)",
			data: untrustedCertCert,
		},
		{
			name: "untrusted cert (trust)",
			data: untrustedCertTrust,
		},
		{
			name: "malformed class",
			data: malformedClass,
			err:  "malformed CKA_CLASS",
		},
		{
			name: "malformed cert octal",
			data: malformedOctalCert,
			err:  "invalid syntax",
		},
		{
			name: "malformed hash octal",
			data: malformedOctalhash,
			err:  "invalid syntax",
		},
		{
			name: "malformed distrust octal",
			data: malformedOctalDistrust,
			err:  "invalid syntax",
		},
	} {
		t.Run(tc.name, func(t *testing.T) {
			r := strings.NewReader(tc.data)
			nc, err := Parse(r)
			if err != nil {
				if tc.err == "" {
					t.Fatalf("unexpected error: %s", err)
				} else if tc.err != err.Error() {
					t.Fatalf("unexpected error: want %q, got %q", tc.err, err)
				}
			} else if tc.err != "" {
				t.Fatal("expected error, got nil")
			}

			if len(tc.output) != len(nc) {
				t.Fatalf("unexpected number of parsed certs: want %d, got %d", len(tc.output), len(nc))
			}

			// sort so we can check equality
			sort.Slice(nc, func(i, j int) bool {
				return nc[i].X509.Subject.String() < nc[j].X509.Subject.String()
			})

			if !reflect.DeepEqual(tc.output, nc) {
				t.Fatal("unexpected results")
			}
		})
	}
}

const validCertdata = `#
# Certificate "Comodo AAA Services root"
#
# Issuer: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Serial Number: 1 (0x1)
# Subject: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Not Valid Before: Thu Jan 01 00:00:00 2004
# Not Valid After : Sun Dec 31 23:59:59 2028
# Fingerprint (SHA-256): D7:A7:A0:FB:5D:7E:27:31:D7:71:E9:48:4E:BC:DE:F7:1D:5F:0C:3E:0A:29:48:78:2B:C8:3E:E0:EA:69:9E:F4
# Fingerprint (SHA1): D1:EB:23:A4:6D:17:D6:8F:D9:25:64:C2:F1:F1:60:17:64:D8:E3:49
CKA_CLASS CK_OBJECT_CLASS CKO_CERTIFICATE
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERTIFICATE_TYPE CK_CERTIFICATE_TYPE CKC_X_509
CKA_SUBJECT MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_ID UTF8 "0"
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_VALUE MULTILINE_OCTAL
\060\202\004\062\060\202\003\032\240\003\002\001\002\002\001\001
\060\015\006\011\052\206\110\206\367\015\001\001\005\005\000\060
\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061\033
\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145\162
\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016\006
\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032\060
\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040\103
\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003\125
\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151\143
\141\164\145\040\123\145\162\166\151\143\145\163\060\036\027\015
\060\064\060\061\060\061\060\060\060\060\060\060\132\027\015\062
\070\061\062\063\061\062\063\065\071\065\071\132\060\173\061\013
\060\011\006\003\125\004\006\023\002\107\102\061\033\060\031\006
\003\125\004\010\014\022\107\162\145\141\164\145\162\040\115\141
\156\143\150\145\163\164\145\162\061\020\060\016\006\003\125\004
\007\014\007\123\141\154\146\157\162\144\061\032\060\030\006\003
\125\004\012\014\021\103\157\155\157\144\157\040\103\101\040\114
\151\155\151\164\145\144\061\041\060\037\006\003\125\004\003\014
\030\101\101\101\040\103\145\162\164\151\146\151\143\141\164\145
\040\123\145\162\166\151\143\145\163\060\202\001\042\060\015\006
\011\052\206\110\206\367\015\001\001\001\005\000\003\202\001\017
\000\060\202\001\012\002\202\001\001\000\276\100\235\364\156\341
\352\166\207\034\115\105\104\216\276\106\310\203\006\235\301\052
\376\030\037\216\344\002\372\363\253\135\120\212\026\061\013\232
\006\320\305\160\042\315\111\055\124\143\314\266\156\150\106\013
\123\352\313\114\044\300\274\162\116\352\361\025\256\364\124\232
\022\012\303\172\262\063\140\342\332\211\125\363\042\130\363\336
\334\317\357\203\206\242\214\224\117\237\150\362\230\220\106\204
\047\307\166\277\343\314\065\054\213\136\007\144\145\202\300\110
\260\250\221\371\141\237\166\040\120\250\221\307\146\265\353\170
\142\003\126\360\212\032\023\352\061\243\036\240\231\375\070\366
\366\047\062\130\157\007\365\153\270\373\024\053\257\267\252\314
\326\143\137\163\214\332\005\231\250\070\250\313\027\170\066\121
\254\351\236\364\170\072\215\317\017\331\102\342\230\014\253\057
\237\016\001\336\357\237\231\111\361\055\337\254\164\115\033\230
\265\107\305\345\051\321\371\220\030\307\142\234\276\203\307\046
\173\076\212\045\307\300\335\235\346\065\150\020\040\235\217\330
\336\322\303\204\234\015\136\350\057\311\002\003\001\000\001\243
\201\300\060\201\275\060\035\006\003\125\035\016\004\026\004\024
\240\021\012\043\076\226\361\007\354\342\257\051\357\202\245\177
\320\060\244\264\060\016\006\003\125\035\017\001\001\377\004\004
\003\002\001\006\060\017\006\003\125\035\023\001\001\377\004\005
\060\003\001\001\377\060\173\006\003\125\035\037\004\164\060\162
\060\070\240\066\240\064\206\062\150\164\164\160\072\057\057\143
\162\154\056\143\157\155\157\144\157\143\141\056\143\157\155\057
\101\101\101\103\145\162\164\151\146\151\143\141\164\145\123\145
\162\166\151\143\145\163\056\143\162\154\060\066\240\064\240\062
\206\060\150\164\164\160\072\057\057\143\162\154\056\143\157\155
\157\144\157\056\156\145\164\057\101\101\101\103\145\162\164\151
\146\151\143\141\164\145\123\145\162\166\151\143\145\163\056\143
\162\154\060\015\006\011\052\206\110\206\367\015\001\001\005\005
\000\003\202\001\001\000\010\126\374\002\360\233\350\377\244\372
\326\173\306\104\200\316\117\304\305\366\000\130\314\246\266\274
\024\111\150\004\166\350\346\356\135\354\002\017\140\326\215\120
\030\117\046\116\001\343\346\260\245\356\277\274\164\124\101\277
\375\374\022\270\307\117\132\364\211\140\005\177\140\267\005\112
\363\366\361\302\277\304\271\164\206\266\055\175\153\314\322\363
\106\335\057\306\340\152\303\303\064\003\054\175\226\335\132\302
\016\247\012\231\301\005\213\253\014\057\363\134\072\317\154\067
\125\011\207\336\123\100\154\130\357\374\266\253\145\156\004\366
\033\334\074\340\132\025\306\236\331\361\131\110\060\041\145\003
\154\354\351\041\163\354\233\003\241\340\067\255\240\025\030\217
\372\272\002\316\247\054\251\020\023\054\324\345\010\046\253\042
\227\140\370\220\136\164\324\242\232\123\275\362\251\150\340\242
\156\302\327\154\261\243\017\236\277\353\150\347\126\362\256\362
\343\053\070\072\011\201\265\153\205\327\276\055\355\077\032\267
\262\143\342\365\142\054\202\324\152\000\101\120\361\071\203\237
\225\351\066\226\230\156
END
CKA_NSS_MOZILLA_CA_POLICY CK_BBOOL CK_TRUE
CKA_NSS_SERVER_DISTRUST_AFTER CK_BBOOL CK_FALSE
CKA_NSS_EMAIL_DISTRUST_AFTER CK_BBOOL CK_FALSE

# Trust for "Comodo AAA Services root"
# Issuer: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Serial Number: 1 (0x1)
# Subject: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Not Valid Before: Thu Jan 01 00:00:00 2004
# Not Valid After : Sun Dec 31 23:59:59 2028
# Fingerprint (SHA-256): D7:A7:A0:FB:5D:7E:27:31:D7:71:E9:48:4E:BC:DE:F7:1D:5F:0C:3E:0A:29:48:78:2B:C8:3E:E0:EA:69:9E:F4
# Fingerprint (SHA1): D1:EB:23:A4:6D:17:D6:8F:D9:25:64:C2:F1:F1:60:17:64:D8:E3:49
CKA_CLASS CK_OBJECT_CLASS CKO_NSS_TRUST
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERT_SHA1_HASH MULTILINE_OCTAL
\321\353\043\244\155\027\326\217\331\045\144\302\361\361\140\027
\144\330\343\111
END
CKA_CERT_MD5_HASH MULTILINE_OCTAL
\111\171\004\260\353\207\031\254\107\260\274\021\121\233\164\320
END
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_TRUST_SERVER_AUTH CK_TRUST CKT_NSS_TRUSTED_DELEGATOR
CKA_TRUST_EMAIL_PROTECTION CK_TRUST CKT_NSS_TRUSTED_DELEGATOR
CKA_TRUST_CODE_SIGNING CK_TRUST CKT_NSS_MUST_VERIFY_TRUST
CKA_TRUST_STEP_UP_APPROVED CK_BBOOL CK_FALSE

#
# Certificate "TrustCor RootCert CA-2"
#
# Issuer: CN=TrustCor RootCert CA-2,OU=TrustCor Certificate Authority,O=TrustCor Systems S. de R.L.,L=Panama City,ST=Panama,C=PA
# Serial Number:25:a1:df:ca:33:cb:59:02
# Subject: CN=TrustCor RootCert CA-2,OU=TrustCor Certificate Authority,O=TrustCor Systems S. de R.L.,L=Panama City,ST=Panama,C=PA
# Not Valid Before: Thu Feb 04 12:32:23 2016
# Not Valid After : Sun Dec 31 17:26:39 2034
# Fingerprint (SHA-256): 07:53:E9:40:37:8C:1B:D5:E3:83:6E:39:5D:AE:A5:CB:83:9E:50:46:F1:BD:0E:AE:19:51:CF:10:FE:C7:C9:65
# Fingerprint (SHA1): B8:BE:6D:CB:56:F1:55:B9:63:D4:12:CA:4E:06:34:C7:94:B2:1C:C0
CKA_CLASS CK_OBJECT_CLASS CKO_CERTIFICATE
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "TrustCor RootCert CA-2"
CKA_CERTIFICATE_TYPE CK_CERTIFICATE_TYPE CKC_X_509
CKA_SUBJECT MULTILINE_OCTAL
\060\201\244\061\013\060\011\006\003\125\004\006\023\002\120\101
\061\017\060\015\006\003\125\004\010\014\006\120\141\156\141\155
\141\061\024\060\022\006\003\125\004\007\014\013\120\141\156\141
\155\141\040\103\151\164\171\061\044\060\042\006\003\125\004\012
\014\033\124\162\165\163\164\103\157\162\040\123\171\163\164\145
\155\163\040\123\056\040\144\145\040\122\056\114\056\061\047\060
\045\006\003\125\004\013\014\036\124\162\165\163\164\103\157\162
\040\103\145\162\164\151\146\151\143\141\164\145\040\101\165\164
\150\157\162\151\164\171\061\037\060\035\006\003\125\004\003\014
\026\124\162\165\163\164\103\157\162\040\122\157\157\164\103\145
\162\164\040\103\101\055\062
END
CKA_ID UTF8 "0"
CKA_ISSUER MULTILINE_OCTAL
\060\201\244\061\013\060\011\006\003\125\004\006\023\002\120\101
\061\017\060\015\006\003\125\004\010\014\006\120\141\156\141\155
\141\061\024\060\022\006\003\125\004\007\014\013\120\141\156\141
\155\141\040\103\151\164\171\061\044\060\042\006\003\125\004\012
\014\033\124\162\165\163\164\103\157\162\040\123\171\163\164\145
\155\163\040\123\056\040\144\145\040\122\056\114\056\061\047\060
\045\006\003\125\004\013\014\036\124\162\165\163\164\103\157\162
\040\103\145\162\164\151\146\151\143\141\164\145\040\101\165\164
\150\157\162\151\164\171\061\037\060\035\006\003\125\004\003\014
\026\124\162\165\163\164\103\157\162\040\122\157\157\164\103\145
\162\164\040\103\101\055\062
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\010\045\241\337\312\063\313\131\002
END
CKA_VALUE MULTILINE_OCTAL
\060\202\006\057\060\202\004\027\240\003\002\001\002\002\010\045
\241\337\312\063\313\131\002\060\015\006\011\052\206\110\206\367
\015\001\001\013\005\000\060\201\244\061\013\060\011\006\003\125
\004\006\023\002\120\101\061\017\060\015\006\003\125\004\010\014
\006\120\141\156\141\155\141\061\024\060\022\006\003\125\004\007
\014\013\120\141\156\141\155\141\040\103\151\164\171\061\044\060
\042\006\003\125\004\012\014\033\124\162\165\163\164\103\157\162
\040\123\171\163\164\145\155\163\040\123\056\040\144\145\040\122
\056\114\056\061\047\060\045\006\003\125\004\013\014\036\124\162
\165\163\164\103\157\162\040\103\145\162\164\151\146\151\143\141
\164\145\040\101\165\164\150\157\162\151\164\171\061\037\060\035
\006\003\125\004\003\014\026\124\162\165\163\164\103\157\162\040
\122\157\157\164\103\145\162\164\040\103\101\055\062\060\036\027
\015\061\066\060\062\060\064\061\062\063\062\062\063\132\027\015
\063\064\061\062\063\061\061\067\062\066\063\071\132\060\201\244
\061\013\060\011\006\003\125\004\006\023\002\120\101\061\017\060
\015\006\003\125\004\010\014\006\120\141\156\141\155\141\061\024
\060\022\006\003\125\004\007\014\013\120\141\156\141\155\141\040
\103\151\164\171\061\044\060\042\006\003\125\004\012\014\033\124
\162\165\163\164\103\157\162\040\123\171\163\164\145\155\163\040
\123\056\040\144\145\040\122\056\114\056\061\047\060\045\006\003
\125\004\013\014\036\124\162\165\163\164\103\157\162\040\103\145
\162\164\151\146\151\143\141\164\145\040\101\165\164\150\157\162
\151\164\171\061\037\060\035\006\003\125\004\003\014\026\124\162
\165\163\164\103\157\162\040\122\157\157\164\103\145\162\164\040
\103\101\055\062\060\202\002\042\060\015\006\011\052\206\110\206
\367\015\001\001\001\005\000\003\202\002\017\000\060\202\002\012
\002\202\002\001\000\247\040\156\302\052\242\142\044\225\220\166
\310\070\176\200\322\253\301\233\145\005\224\364\301\012\020\325
\002\254\355\237\223\307\207\310\260\047\053\102\014\075\012\076
\101\132\236\165\335\215\312\340\233\354\150\062\244\151\222\150
\214\013\201\016\126\240\076\032\335\054\045\024\202\057\227\323
\144\106\364\124\251\334\072\124\055\061\053\231\202\362\331\052
\327\357\161\000\270\061\244\276\172\044\007\303\102\040\362\212
\324\222\004\033\145\126\114\154\324\373\266\141\132\107\043\264
\330\151\264\267\072\320\164\074\014\165\241\214\116\166\241\351
\333\052\245\073\372\316\260\377\176\152\050\375\047\034\310\261
\351\051\361\127\156\144\264\320\301\025\155\016\276\056\016\106
\310\136\364\121\376\357\016\143\072\073\161\272\317\157\131\312
\014\343\233\135\111\270\114\342\127\261\230\212\102\127\234\166
\357\357\275\321\150\250\322\364\011\273\167\065\276\045\202\010
\304\026\054\104\040\126\251\104\021\167\357\135\264\035\252\136
\153\076\213\062\366\007\057\127\004\222\312\365\376\235\302\351
\350\263\216\114\113\002\061\331\344\074\110\202\047\367\030\202
\166\110\072\161\261\023\241\071\325\056\305\064\302\035\142\205
\337\003\376\115\364\257\075\337\134\133\215\372\160\341\245\176
\047\307\206\056\152\217\022\306\204\136\103\121\120\234\031\233
\170\346\374\366\355\107\176\173\075\146\357\023\023\210\137\074
\241\143\373\371\254\207\065\237\363\202\236\244\077\012\234\061
\151\213\231\244\210\112\216\156\146\115\357\026\304\017\171\050
\041\140\015\205\026\175\327\124\070\361\222\126\375\265\063\114
\203\334\327\020\237\113\375\306\370\102\275\272\174\163\002\340
\377\175\315\133\341\324\254\141\173\127\325\112\173\133\324\205
\130\047\135\277\370\053\140\254\240\046\256\024\041\047\306\167
\232\063\200\074\136\106\077\367\303\261\243\206\063\306\350\136
\015\271\065\054\252\106\301\205\002\165\200\240\353\044\373\025
\252\344\147\177\156\167\077\364\004\212\057\174\173\343\027\141
\360\335\011\251\040\310\276\011\244\320\176\104\303\262\060\112
\070\252\251\354\030\232\007\202\053\333\270\234\030\255\332\340
\106\027\254\317\135\002\003\001\000\001\243\143\060\141\060\035
\006\003\125\035\016\004\026\004\024\331\376\041\100\156\224\236
\274\233\075\234\175\230\040\031\345\214\060\142\262\060\037\006
\003\125\035\043\004\030\060\026\200\024\331\376\041\100\156\224
\236\274\233\075\234\175\230\040\031\345\214\060\142\262\060\017
\006\003\125\035\023\001\001\377\004\005\060\003\001\001\377\060
\016\006\003\125\035\017\001\001\377\004\004\003\002\001\206\060
\015\006\011\052\206\110\206\367\015\001\001\013\005\000\003\202
\002\001\000\236\105\236\014\073\266\357\341\072\310\174\321\000
\075\317\342\352\006\265\262\072\273\006\113\150\172\320\043\227
\164\247\054\360\010\330\171\132\327\132\204\212\330\022\232\033
\331\175\134\115\160\305\245\371\253\345\243\211\211\335\001\372
\354\335\371\351\222\227\333\260\106\102\363\323\142\252\225\376
\061\147\024\151\130\220\012\252\013\356\067\043\307\120\121\264
\365\176\236\343\173\367\344\314\102\062\055\111\014\313\377\111
\014\233\036\064\375\156\156\226\212\171\003\266\157\333\011\313
\375\137\145\024\067\341\070\365\363\141\026\130\344\265\155\015
\013\004\033\077\120\055\177\263\307\172\032\026\200\140\370\212
\037\351\033\052\306\371\272\001\032\151\277\322\130\307\124\127
\010\217\341\071\140\167\113\254\131\204\032\210\361\335\313\117
\170\327\347\341\063\055\374\356\101\372\040\260\276\313\367\070
\224\300\341\320\205\017\273\355\054\163\253\355\376\222\166\032
\144\177\133\015\063\011\007\063\173\006\077\021\244\134\160\074
\205\300\317\343\220\250\203\167\372\333\346\305\214\150\147\020
\147\245\122\055\360\304\231\217\177\277\321\153\342\265\107\326
\331\320\205\231\115\224\233\017\113\215\356\000\132\107\035\021
\003\254\101\030\257\207\267\157\014\072\217\312\317\334\003\301
\242\011\310\345\375\200\136\310\140\102\001\033\032\123\132\273
\067\246\267\274\272\204\351\036\154\032\324\144\332\324\103\376
\223\213\113\362\054\171\026\020\324\223\013\210\217\241\330\206
\024\106\221\107\233\050\044\357\127\122\116\134\102\234\252\367
\111\354\047\350\100\036\263\246\211\042\162\234\365\015\063\264
\130\243\060\073\335\324\152\124\223\276\032\115\363\223\224\367
\374\204\013\077\204\040\134\064\003\104\305\332\255\274\012\301
\002\317\036\345\224\331\363\216\133\330\114\360\235\354\141\027
\273\024\062\124\014\002\051\223\036\222\206\366\177\357\347\222
\005\016\131\335\231\010\056\056\372\234\000\122\323\305\146\051
\344\247\227\104\244\016\050\201\023\065\305\366\157\144\346\101
\304\325\057\314\064\105\045\317\101\000\226\075\112\056\302\226
\230\117\116\112\234\227\267\333\037\222\062\310\377\017\121\156
\326\354\011
END
CKA_NSS_MOZILLA_CA_POLICY CK_BBOOL CK_TRUE
# For Server Distrust After: Wed Nov 30 00:00:00 2022
CKA_NSS_SERVER_DISTRUST_AFTER MULTILINE_OCTAL
\062\062\061\061\063\060\060\060\060\060\060\060\132
END
# For Email Distrust After: Wed Nov 30 00:00:00 2022
CKA_NSS_EMAIL_DISTRUST_AFTER MULTILINE_OCTAL
\062\062\061\061\063\060\060\060\060\060\060\060\132
END


# Trust for "TrustCor RootCert CA-2"
# Issuer: CN=TrustCor RootCert CA-2,OU=TrustCor Certificate Authority,O=TrustCor Systems S. de R.L.,L=Panama City,ST=Panama,C=PA
# Serial Number:25:a1:df:ca:33:cb:59:02
# Subject: CN=TrustCor RootCert CA-2,OU=TrustCor Certificate Authority,O=TrustCor Systems S. de R.L.,L=Panama City,ST=Panama,C=PA
# Not Valid Before: Thu Feb 04 12:32:23 2016
# Not Valid After : Sun Dec 31 17:26:39 2034
# Fingerprint (SHA-256): 07:53:E9:40:37:8C:1B:D5:E3:83:6E:39:5D:AE:A5:CB:83:9E:50:46:F1:BD:0E:AE:19:51:CF:10:FE:C7:C9:65
# Fingerprint (SHA1): B8:BE:6D:CB:56:F1:55:B9:63:D4:12:CA:4E:06:34:C7:94:B2:1C:C0
CKA_CLASS CK_OBJECT_CLASS CKO_NSS_TRUST
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "TrustCor RootCert CA-2"
CKA_CERT_SHA1_HASH MULTILINE_OCTAL
\270\276\155\313\126\361\125\271\143\324\022\312\116\006\064\307
\224\262\034\300
END
CKA_CERT_MD5_HASH MULTILINE_OCTAL
\242\341\370\030\013\272\105\325\307\101\052\273\067\122\105\144
END
CKA_ISSUER MULTILINE_OCTAL
\060\201\244\061\013\060\011\006\003\125\004\006\023\002\120\101
\061\017\060\015\006\003\125\004\010\014\006\120\141\156\141\155
\141\061\024\060\022\006\003\125\004\007\014\013\120\141\156\141
\155\141\040\103\151\164\171\061\044\060\042\006\003\125\004\012
\014\033\124\162\165\163\164\103\157\162\040\123\171\163\164\145
\155\163\040\123\056\040\144\145\040\122\056\114\056\061\047\060
\045\006\003\125\004\013\014\036\124\162\165\163\164\103\157\162
\040\103\145\162\164\151\146\151\143\141\164\145\040\101\165\164
\150\157\162\151\164\171\061\037\060\035\006\003\125\004\003\014
\026\124\162\165\163\164\103\157\162\040\122\157\157\164\103\145
\162\164\040\103\101\055\062
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\010\045\241\337\312\063\313\131\002
END
CKA_TRUST_SERVER_AUTH CK_TRUST CKT_NSS_TRUSTED_DELEGATOR
CKA_TRUST_EMAIL_PROTECTION CK_TRUST CKT_NSS_TRUSTED_DELEGATOR
CKA_TRUST_CODE_SIGNING CK_TRUST CKT_NSS_MUST_VERIFY_TRUST
CKA_TRUST_STEP_UP_APPROVED CK_BBOOL CK_FALSE
`

const trustNoCert = `CKA_CLASS CK_OBJECT_CLASS CKO_NSS_TRUST
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERT_SHA1_HASH MULTILINE_OCTAL
\321\353\043\244\155\027\326\217\331\045\144\302\361\361\140\027
\144\330\343\111
END
CKA_CERT_MD5_HASH MULTILINE_OCTAL
\111\171\004\260\353\207\031\254\107\260\274\021\121\233\164\320
END
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_TRUST_SERVER_AUTH CK_TRUST CKT_NSS_TRUSTED_DELEGATOR
CKA_TRUST_EMAIL_PROTECTION CK_TRUST CKT_NSS_TRUSTED_DELEGATOR
CKA_TRUST_CODE_SIGNING CK_TRUST CKT_NSS_MUST_VERIFY_TRUST
CKA_TRUST_STEP_UP_APPROVED CK_BBOOL CK_FALSE`

const certNoTrust = `CKA_CLASS CK_OBJECT_CLASS CKO_CERTIFICATE
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERTIFICATE_TYPE CK_CERTIFICATE_TYPE CKC_X_509
CKA_SUBJECT MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_ID UTF8 "0"
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_VALUE MULTILINE_OCTAL
\060\202\004\062\060\202\003\032\240\003\002\001\002\002\001\001
\060\015\006\011\052\206\110\206\367\015\001\001\005\005\000\060
\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061\033
\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145\162
\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016\006
\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032\060
\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040\103
\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003\125
\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151\143
\141\164\145\040\123\145\162\166\151\143\145\163\060\036\027\015
\060\064\060\061\060\061\060\060\060\060\060\060\132\027\015\062
\070\061\062\063\061\062\063\065\071\065\071\132\060\173\061\013
\060\011\006\003\125\004\006\023\002\107\102\061\033\060\031\006
\003\125\004\010\014\022\107\162\145\141\164\145\162\040\115\141
\156\143\150\145\163\164\145\162\061\020\060\016\006\003\125\004
\007\014\007\123\141\154\146\157\162\144\061\032\060\030\006\003
\125\004\012\014\021\103\157\155\157\144\157\040\103\101\040\114
\151\155\151\164\145\144\061\041\060\037\006\003\125\004\003\014
\030\101\101\101\040\103\145\162\164\151\146\151\143\141\164\145
\040\123\145\162\166\151\143\145\163\060\202\001\042\060\015\006
\011\052\206\110\206\367\015\001\001\001\005\000\003\202\001\017
\000\060\202\001\012\002\202\001\001\000\276\100\235\364\156\341
\352\166\207\034\115\105\104\216\276\106\310\203\006\235\301\052
\376\030\037\216\344\002\372\363\253\135\120\212\026\061\013\232
\006\320\305\160\042\315\111\055\124\143\314\266\156\150\106\013
\123\352\313\114\044\300\274\162\116\352\361\025\256\364\124\232
\022\012\303\172\262\063\140\342\332\211\125\363\042\130\363\336
\334\317\357\203\206\242\214\224\117\237\150\362\230\220\106\204
\047\307\166\277\343\314\065\054\213\136\007\144\145\202\300\110
\260\250\221\371\141\237\166\040\120\250\221\307\146\265\353\170
\142\003\126\360\212\032\023\352\061\243\036\240\231\375\070\366
\366\047\062\130\157\007\365\153\270\373\024\053\257\267\252\314
\326\143\137\163\214\332\005\231\250\070\250\313\027\170\066\121
\254\351\236\364\170\072\215\317\017\331\102\342\230\014\253\057
\237\016\001\336\357\237\231\111\361\055\337\254\164\115\033\230
\265\107\305\345\051\321\371\220\030\307\142\234\276\203\307\046
\173\076\212\045\307\300\335\235\346\065\150\020\040\235\217\330
\336\322\303\204\234\015\136\350\057\311\002\003\001\000\001\243
\201\300\060\201\275\060\035\006\003\125\035\016\004\026\004\024
\240\021\012\043\076\226\361\007\354\342\257\051\357\202\245\177
\320\060\244\264\060\016\006\003\125\035\017\001\001\377\004\004
\003\002\001\006\060\017\006\003\125\035\023\001\001\377\004\005
\060\003\001\001\377\060\173\006\003\125\035\037\004\164\060\162
\060\070\240\066\240\064\206\062\150\164\164\160\072\057\057\143
\162\154\056\143\157\155\157\144\157\143\141\056\143\157\155\057
\101\101\101\103\145\162\164\151\146\151\143\141\164\145\123\145
\162\166\151\143\145\163\056\143\162\154\060\066\240\064\240\062
\206\060\150\164\164\160\072\057\057\143\162\154\056\143\157\155
\157\144\157\056\156\145\164\057\101\101\101\103\145\162\164\151
\146\151\143\141\164\145\123\145\162\166\151\143\145\163\056\143
\162\154\060\015\006\011\052\206\110\206\367\015\001\001\005\005
\000\003\202\001\001\000\010\126\374\002\360\233\350\377\244\372
\326\173\306\104\200\316\117\304\305\366\000\130\314\246\266\274
\024\111\150\004\166\350\346\356\135\354\002\017\140\326\215\120
\030\117\046\116\001\343\346\260\245\356\277\274\164\124\101\277
\375\374\022\270\307\117\132\364\211\140\005\177\140\267\005\112
\363\366\361\302\277\304\271\164\206\266\055\175\153\314\322\363
\106\335\057\306\340\152\303\303\064\003\054\175\226\335\132\302
\016\247\012\231\301\005\213\253\014\057\363\134\072\317\154\067
\125\011\207\336\123\100\154\130\357\374\266\253\145\156\004\366
\033\334\074\340\132\025\306\236\331\361\131\110\060\041\145\003
\154\354\351\041\163\354\233\003\241\340\067\255\240\025\030\217
\372\272\002\316\247\054\251\020\023\054\324\345\010\046\253\042
\227\140\370\220\136\164\324\242\232\123\275\362\251\150\340\242
\156\302\327\154\261\243\017\236\277\353\150\347\126\362\256\362
\343\053\070\072\011\201\265\153\205\327\276\055\355\077\032\267
\262\143\342\365\142\054\202\324\152\000\101\120\361\071\203\237
\225\351\066\226\230\156
END
CKA_NSS_MOZILLA_CA_POLICY CK_BBOOL CK_TRUE
CKA_NSS_SERVER_DISTRUST_AFTER CK_BBOOL CK_FALSE
CKA_NSS_EMAIL_DISTRUST_AFTER CK_BBOOL CK_FALSE`

const missingCertificateOctal = `CKA_CLASS CK_OBJECT_CLASS CKO_CERTIFICATE`

const untrustedCertCert = `#
# Certificate "Comodo AAA Services root"
#
# Issuer: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Serial Number: 1 (0x1)
# Subject: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Not Valid Before: Thu Jan 01 00:00:00 2004
# Not Valid After : Sun Dec 31 23:59:59 2028
# Fingerprint (SHA-256): D7:A7:A0:FB:5D:7E:27:31:D7:71:E9:48:4E:BC:DE:F7:1D:5F:0C:3E:0A:29:48:78:2B:C8:3E:E0:EA:69:9E:F4
# Fingerprint (SHA1): D1:EB:23:A4:6D:17:D6:8F:D9:25:64:C2:F1:F1:60:17:64:D8:E3:49
CKA_CLASS CK_OBJECT_CLASS CKO_CERTIFICATE
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERTIFICATE_TYPE CK_CERTIFICATE_TYPE CKC_X_509
CKA_SUBJECT MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_ID UTF8 "0"
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_VALUE MULTILINE_OCTAL
\060\202\004\062\060\202\003\032\240\003\002\001\002\002\001\001
\060\015\006\011\052\206\110\206\367\015\001\001\005\005\000\060
\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061\033
\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145\162
\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016\006
\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032\060
\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040\103
\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003\125
\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151\143
\141\164\145\040\123\145\162\166\151\143\145\163\060\036\027\015
\060\064\060\061\060\061\060\060\060\060\060\060\132\027\015\062
\070\061\062\063\061\062\063\065\071\065\071\132\060\173\061\013
\060\011\006\003\125\004\006\023\002\107\102\061\033\060\031\006
\003\125\004\010\014\022\107\162\145\141\164\145\162\040\115\141
\156\143\150\145\163\164\145\162\061\020\060\016\006\003\125\004
\007\014\007\123\141\154\146\157\162\144\061\032\060\030\006\003
\125\004\012\014\021\103\157\155\157\144\157\040\103\101\040\114
\151\155\151\164\145\144\061\041\060\037\006\003\125\004\003\014
\030\101\101\101\040\103\145\162\164\151\146\151\143\141\164\145
\040\123\145\162\166\151\143\145\163\060\202\001\042\060\015\006
\011\052\206\110\206\367\015\001\001\001\005\000\003\202\001\017
\000\060\202\001\012\002\202\001\001\000\276\100\235\364\156\341
\352\166\207\034\115\105\104\216\276\106\310\203\006\235\301\052
\376\030\037\216\344\002\372\363\253\135\120\212\026\061\013\232
\006\320\305\160\042\315\111\055\124\143\314\266\156\150\106\013
\123\352\313\114\044\300\274\162\116\352\361\025\256\364\124\232
\022\012\303\172\262\063\140\342\332\211\125\363\042\130\363\336
\334\317\357\203\206\242\214\224\117\237\150\362\230\220\106\204
\047\307\166\277\343\314\065\054\213\136\007\144\145\202\300\110
\260\250\221\371\141\237\166\040\120\250\221\307\146\265\353\170
\142\003\126\360\212\032\023\352\061\243\036\240\231\375\070\366
\366\047\062\130\157\007\365\153\270\373\024\053\257\267\252\314
\326\143\137\163\214\332\005\231\250\070\250\313\027\170\066\121
\254\351\236\364\170\072\215\317\017\331\102\342\230\014\253\057
\237\016\001\336\357\237\231\111\361\055\337\254\164\115\033\230
\265\107\305\345\051\321\371\220\030\307\142\234\276\203\307\046
\173\076\212\045\307\300\335\235\346\065\150\020\040\235\217\330
\336\322\303\204\234\015\136\350\057\311\002\003\001\000\001\243
\201\300\060\201\275\060\035\006\003\125\035\016\004\026\004\024
\240\021\012\043\076\226\361\007\354\342\257\051\357\202\245\177
\320\060\244\264\060\016\006\003\125\035\017\001\001\377\004\004
\003\002\001\006\060\017\006\003\125\035\023\001\001\377\004\005
\060\003\001\001\377\060\173\006\003\125\035\037\004\164\060\162
\060\070\240\066\240\064\206\062\150\164\164\160\072\057\057\143
\162\154\056\143\157\155\157\144\157\143\141\056\143\157\155\057
\101\101\101\103\145\162\164\151\146\151\143\141\164\145\123\145
\162\166\151\143\145\163\056\143\162\154\060\066\240\064\240\062
\206\060\150\164\164\160\072\057\057\143\162\154\056\143\157\155
\157\144\157\056\156\145\164\057\101\101\101\103\145\162\164\151
\146\151\143\141\164\145\123\145\162\166\151\143\145\163\056\143
\162\154\060\015\006\011\052\206\110\206\367\015\001\001\005\005
\000\003\202\001\001\000\010\126\374\002\360\233\350\377\244\372
\326\173\306\104\200\316\117\304\305\366\000\130\314\246\266\274
\024\111\150\004\166\350\346\356\135\354\002\017\140\326\215\120
\030\117\046\116\001\343\346\260\245\356\277\274\164\124\101\277
\375\374\022\270\307\117\132\364\211\140\005\177\140\267\005\112
\363\366\361\302\277\304\271\164\206\266\055\175\153\314\322\363
\106\335\057\306\340\152\303\303\064\003\054\175\226\335\132\302
\016\247\012\231\301\005\213\253\014\057\363\134\072\317\154\067
\125\011\207\336\123\100\154\130\357\374\266\253\145\156\004\366
\033\334\074\340\132\025\306\236\331\361\131\110\060\041\145\003
\154\354\351\041\163\354\233\003\241\340\067\255\240\025\030\217
\372\272\002\316\247\054\251\020\023\054\324\345\010\046\253\042
\227\140\370\220\136\164\324\242\232\123\275\362\251\150\340\242
\156\302\327\154\261\243\017\236\277\353\150\347\126\362\256\362
\343\053\070\072\011\201\265\153\205\327\276\055\355\077\032\267
\262\143\342\365\142\054\202\324\152\000\101\120\361\071\203\237
\225\351\066\226\230\156
END
CKA_NSS_MOZILLA_CA_POLICY CK_BBOOL CK_FALSE
CKA_NSS_SERVER_DISTRUST_AFTER CK_BBOOL CK_FALSE
CKA_NSS_EMAIL_DISTRUST_AFTER CK_BBOOL CK_FALSE

# Trust for "Comodo AAA Services root"
# Issuer: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Serial Number: 1 (0x1)
# Subject: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Not Valid Before: Thu Jan 01 00:00:00 2004
# Not Valid After : Sun Dec 31 23:59:59 2028
# Fingerprint (SHA-256): D7:A7:A0:FB:5D:7E:27:31:D7:71:E9:48:4E:BC:DE:F7:1D:5F:0C:3E:0A:29:48:78:2B:C8:3E:E0:EA:69:9E:F4
# Fingerprint (SHA1): D1:EB:23:A4:6D:17:D6:8F:D9:25:64:C2:F1:F1:60:17:64:D8:E3:49
CKA_CLASS CK_OBJECT_CLASS CKO_NSS_TRUST
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERT_SHA1_HASH MULTILINE_OCTAL
\321\353\043\244\155\027\326\217\331\045\144\302\361\361\140\027
\144\330\343\111
END
CKA_CERT_MD5_HASH MULTILINE_OCTAL
\111\171\004\260\353\207\031\254\107\260\274\021\121\233\164\320
END
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_TRUST_SERVER_AUTH CK_TRUST CKT_NSS_TRUSTED_DELEGATOR
CKA_TRUST_EMAIL_PROTECTION CK_TRUST CKT_NSS_TRUSTED_DELEGATOR
CKA_TRUST_CODE_SIGNING CK_TRUST CKT_NSS_MUST_VERIFY_TRUST
CKA_TRUST_STEP_UP_APPROVED CK_BBOOL CK_FALSE`

const untrustedCertTrust = `#
# Certificate "Comodo AAA Services root"
#
# Issuer: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Serial Number: 1 (0x1)
# Subject: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Not Valid Before: Thu Jan 01 00:00:00 2004
# Not Valid After : Sun Dec 31 23:59:59 2028
# Fingerprint (SHA-256): D7:A7:A0:FB:5D:7E:27:31:D7:71:E9:48:4E:BC:DE:F7:1D:5F:0C:3E:0A:29:48:78:2B:C8:3E:E0:EA:69:9E:F4
# Fingerprint (SHA1): D1:EB:23:A4:6D:17:D6:8F:D9:25:64:C2:F1:F1:60:17:64:D8:E3:49
CKA_CLASS CK_OBJECT_CLASS CKO_CERTIFICATE
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERTIFICATE_TYPE CK_CERTIFICATE_TYPE CKC_X_509
CKA_SUBJECT MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_ID UTF8 "0"
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_VALUE MULTILINE_OCTAL
\060\202\004\062\060\202\003\032\240\003\002\001\002\002\001\001
\060\015\006\011\052\206\110\206\367\015\001\001\005\005\000\060
\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061\033
\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145\162
\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016\006
\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032\060
\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040\103
\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003\125
\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151\143
\141\164\145\040\123\145\162\166\151\143\145\163\060\036\027\015
\060\064\060\061\060\061\060\060\060\060\060\060\132\027\015\062
\070\061\062\063\061\062\063\065\071\065\071\132\060\173\061\013
\060\011\006\003\125\004\006\023\002\107\102\061\033\060\031\006
\003\125\004\010\014\022\107\162\145\141\164\145\162\040\115\141
\156\143\150\145\163\164\145\162\061\020\060\016\006\003\125\004
\007\014\007\123\141\154\146\157\162\144\061\032\060\030\006\003
\125\004\012\014\021\103\157\155\157\144\157\040\103\101\040\114
\151\155\151\164\145\144\061\041\060\037\006\003\125\004\003\014
\030\101\101\101\040\103\145\162\164\151\146\151\143\141\164\145
\040\123\145\162\166\151\143\145\163\060\202\001\042\060\015\006
\011\052\206\110\206\367\015\001\001\001\005\000\003\202\001\017
\000\060\202\001\012\002\202\001\001\000\276\100\235\364\156\341
\352\166\207\034\115\105\104\216\276\106\310\203\006\235\301\052
\376\030\037\216\344\002\372\363\253\135\120\212\026\061\013\232
\006\320\305\160\042\315\111\055\124\143\314\266\156\150\106\013
\123\352\313\114\044\300\274\162\116\352\361\025\256\364\124\232
\022\012\303\172\262\063\140\342\332\211\125\363\042\130\363\336
\334\317\357\203\206\242\214\224\117\237\150\362\230\220\106\204
\047\307\166\277\343\314\065\054\213\136\007\144\145\202\300\110
\260\250\221\371\141\237\166\040\120\250\221\307\146\265\353\170
\142\003\126\360\212\032\023\352\061\243\036\240\231\375\070\366
\366\047\062\130\157\007\365\153\270\373\024\053\257\267\252\314
\326\143\137\163\214\332\005\231\250\070\250\313\027\170\066\121
\254\351\236\364\170\072\215\317\017\331\102\342\230\014\253\057
\237\016\001\336\357\237\231\111\361\055\337\254\164\115\033\230
\265\107\305\345\051\321\371\220\030\307\142\234\276\203\307\046
\173\076\212\045\307\300\335\235\346\065\150\020\040\235\217\330
\336\322\303\204\234\015\136\350\057\311\002\003\001\000\001\243
\201\300\060\201\275\060\035\006\003\125\035\016\004\026\004\024
\240\021\012\043\076\226\361\007\354\342\257\051\357\202\245\177
\320\060\244\264\060\016\006\003\125\035\017\001\001\377\004\004
\003\002\001\006\060\017\006\003\125\035\023\001\001\377\004\005
\060\003\001\001\377\060\173\006\003\125\035\037\004\164\060\162
\060\070\240\066\240\064\206\062\150\164\164\160\072\057\057\143
\162\154\056\143\157\155\157\144\157\143\141\056\143\157\155\057
\101\101\101\103\145\162\164\151\146\151\143\141\164\145\123\145
\162\166\151\143\145\163\056\143\162\154\060\066\240\064\240\062
\206\060\150\164\164\160\072\057\057\143\162\154\056\143\157\155
\157\144\157\056\156\145\164\057\101\101\101\103\145\162\164\151
\146\151\143\141\164\145\123\145\162\166\151\143\145\163\056\143
\162\154\060\015\006\011\052\206\110\206\367\015\001\001\005\005
\000\003\202\001\001\000\010\126\374\002\360\233\350\377\244\372
\326\173\306\104\200\316\117\304\305\366\000\130\314\246\266\274
\024\111\150\004\166\350\346\356\135\354\002\017\140\326\215\120
\030\117\046\116\001\343\346\260\245\356\277\274\164\124\101\277
\375\374\022\270\307\117\132\364\211\140\005\177\140\267\005\112
\363\366\361\302\277\304\271\164\206\266\055\175\153\314\322\363
\106\335\057\306\340\152\303\303\064\003\054\175\226\335\132\302
\016\247\012\231\301\005\213\253\014\057\363\134\072\317\154\067
\125\011\207\336\123\100\154\130\357\374\266\253\145\156\004\366
\033\334\074\340\132\025\306\236\331\361\131\110\060\041\145\003
\154\354\351\041\163\354\233\003\241\340\067\255\240\025\030\217
\372\272\002\316\247\054\251\020\023\054\324\345\010\046\253\042
\227\140\370\220\136\164\324\242\232\123\275\362\251\150\340\242
\156\302\327\154\261\243\017\236\277\353\150\347\126\362\256\362
\343\053\070\072\011\201\265\153\205\327\276\055\355\077\032\267
\262\143\342\365\142\054\202\324\152\000\101\120\361\071\203\237
\225\351\066\226\230\156
END
CKA_NSS_MOZILLA_CA_POLICY CK_BBOOL CK_TRUE
CKA_NSS_SERVER_DISTRUST_AFTER CK_BBOOL CK_FALSE
CKA_NSS_EMAIL_DISTRUST_AFTER CK_BBOOL CK_FALSE

# Trust for "Comodo AAA Services root"
# Issuer: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Serial Number: 1 (0x1)
# Subject: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Not Valid Before: Thu Jan 01 00:00:00 2004
# Not Valid After : Sun Dec 31 23:59:59 2028
# Fingerprint (SHA-256): D7:A7:A0:FB:5D:7E:27:31:D7:71:E9:48:4E:BC:DE:F7:1D:5F:0C:3E:0A:29:48:78:2B:C8:3E:E0:EA:69:9E:F4
# Fingerprint (SHA1): D1:EB:23:A4:6D:17:D6:8F:D9:25:64:C2:F1:F1:60:17:64:D8:E3:49
CKA_CLASS CK_OBJECT_CLASS CKO_NSS_TRUST
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERT_SHA1_HASH MULTILINE_OCTAL
\321\353\043\244\155\027\326\217\331\045\144\302\361\361\140\027
\144\330\343\111
END
CKA_CERT_MD5_HASH MULTILINE_OCTAL
\111\171\004\260\353\207\031\254\107\260\274\021\121\233\164\320
END
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_TRUST_SERVER_AUTH CK_TRUST 
CKA_TRUST_EMAIL_PROTECTION CK_TRUST CKT_NSS_MUST_VERIFY_TRUST
CKA_TRUST_CODE_SIGNING CK_TRUST CKT_NSS_MUST_VERIFY_TRUST
CKA_TRUST_STEP_UP_APPROVED CK_BBOOL CK_FALSE`

const malformedClass = `CKA_CLASS`

const malformedOctalCert = `#
# Certificate "Comodo AAA Services root"
#
# Issuer: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Serial Number: 1 (0x1)
# Subject: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Not Valid Before: Thu Jan 01 00:00:00 2004
# Not Valid After : Sun Dec 31 23:59:59 2028
# Fingerprint (SHA-256): D7:A7:A0:FB:5D:7E:27:31:D7:71:E9:48:4E:BC:DE:F7:1D:5F:0C:3E:0A:29:48:78:2B:C8:3E:E0:EA:69:9E:F4
# Fingerprint (SHA1): D1:EB:23:A4:6D:17:D6:8F:D9:25:64:C2:F1:F1:60:17:64:D8:E3:49
CKA_CLASS CK_OBJECT_CLASS CKO_CERTIFICATE
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERTIFICATE_TYPE CK_CERTIFICATE_TYPE CKC_X_509
CKA_SUBJECT MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_ID UTF8 "0"
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_VALUE MULTILINE_OCTAL
hello\xxxxxxx
END
CKA_NSS_MOZILLA_CA_POLICY CK_BBOOL CK_TRUE
CKA_NSS_SERVER_DISTRUST_AFTER CK_BBOOL CK_FALSE
CKA_NSS_EMAIL_DISTRUST_AFTER CK_BBOOL CK_FALSE`

const malformedOctalhash = `# Trust for "Comodo AAA Services root"
# Issuer: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Serial Number: 1 (0x1)
# Subject: CN=AAA Certificate Services,O=Comodo CA Limited,L=Salford,ST=Greater Manchester,C=GB
# Not Valid Before: Thu Jan 01 00:00:00 2004
# Not Valid After : Sun Dec 31 23:59:59 2028
# Fingerprint (SHA-256): D7:A7:A0:FB:5D:7E:27:31:D7:71:E9:48:4E:BC:DE:F7:1D:5F:0C:3E:0A:29:48:78:2B:C8:3E:E0:EA:69:9E:F4
# Fingerprint (SHA1): D1:EB:23:A4:6D:17:D6:8F:D9:25:64:C2:F1:F1:60:17:64:D8:E3:49
CKA_CLASS CK_OBJECT_CLASS CKO_NSS_TRUST
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "Comodo AAA Services root"
CKA_CERT_SHA1_HASH MULTILINE_OCTAL
\xxxxx
END
CKA_CERT_MD5_HASH MULTILINE_OCTAL
\111\171\004\260\353\207\031\254\107\260\274\021\121\233\164\320
END
CKA_ISSUER MULTILINE_OCTAL
\060\173\061\013\060\011\006\003\125\004\006\023\002\107\102\061
\033\060\031\006\003\125\004\010\014\022\107\162\145\141\164\145
\162\040\115\141\156\143\150\145\163\164\145\162\061\020\060\016
\006\003\125\004\007\014\007\123\141\154\146\157\162\144\061\032
\060\030\006\003\125\004\012\014\021\103\157\155\157\144\157\040
\103\101\040\114\151\155\151\164\145\144\061\041\060\037\006\003
\125\004\003\014\030\101\101\101\040\103\145\162\164\151\146\151
\143\141\164\145\040\123\145\162\166\151\143\145\163
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\001\001
END
CKA_TRUST_SERVER_AUTH CK_TRUST 
CKA_TRUST_EMAIL_PROTECTION CK_TRUST CKT_NSS_MUST_VERIFY_TRUST
CKA_TRUST_CODE_SIGNING CK_TRUST CKT_NSS_MUST_VERIFY_TRUST
CKA_TRUST_STEP_UP_APPROVED CK_BBOOL CK_FALSE`

const malformedOctalDistrust = `#
# Certificate "TrustCor RootCert CA-2"
#
# Issuer: CN=TrustCor RootCert CA-2,OU=TrustCor Certificate Authority,O=TrustCor Systems S. de R.L.,L=Panama City,ST=Panama,C=PA
# Serial Number:25:a1:df:ca:33:cb:59:02
# Subject: CN=TrustCor RootCert CA-2,OU=TrustCor Certificate Authority,O=TrustCor Systems S. de R.L.,L=Panama City,ST=Panama,C=PA
# Not Valid Before: Thu Feb 04 12:32:23 2016
# Not Valid After : Sun Dec 31 17:26:39 2034
# Fingerprint (SHA-256): 07:53:E9:40:37:8C:1B:D5:E3:83:6E:39:5D:AE:A5:CB:83:9E:50:46:F1:BD:0E:AE:19:51:CF:10:FE:C7:C9:65
# Fingerprint (SHA1): B8:BE:6D:CB:56:F1:55:B9:63:D4:12:CA:4E:06:34:C7:94:B2:1C:C0
CKA_CLASS CK_OBJECT_CLASS CKO_CERTIFICATE
CKA_TOKEN CK_BBOOL CK_TRUE
CKA_PRIVATE CK_BBOOL CK_FALSE
CKA_MODIFIABLE CK_BBOOL CK_FALSE
CKA_LABEL UTF8 "TrustCor RootCert CA-2"
CKA_CERTIFICATE_TYPE CK_CERTIFICATE_TYPE CKC_X_509
CKA_SUBJECT MULTILINE_OCTAL
\060\201\244\061\013\060\011\006\003\125\004\006\023\002\120\101
\061\017\060\015\006\003\125\004\010\014\006\120\141\156\141\155
\141\061\024\060\022\006\003\125\004\007\014\013\120\141\156\141
\155\141\040\103\151\164\171\061\044\060\042\006\003\125\004\012
\014\033\124\162\165\163\164\103\157\162\040\123\171\163\164\145
\155\163\040\123\056\040\144\145\040\122\056\114\056\061\047\060
\045\006\003\125\004\013\014\036\124\162\165\163\164\103\157\162
\040\103\145\162\164\151\146\151\143\141\164\145\040\101\165\164
\150\157\162\151\164\171\061\037\060\035\006\003\125\004\003\014
\026\124\162\165\163\164\103\157\162\040\122\157\157\164\103\145
\162\164\040\103\101\055\062
END
CKA_ID UTF8 "0"
CKA_ISSUER MULTILINE_OCTAL
\060\201\244\061\013\060\011\006\003\125\004\006\023\002\120\101
\061\017\060\015\006\003\125\004\010\014\006\120\141\156\141\155
\141\061\024\060\022\006\003\125\004\007\014\013\120\141\156\141
\155\141\040\103\151\164\171\061\044\060\042\006\003\125\004\012
\014\033\124\162\165\163\164\103\157\162\040\123\171\163\164\145
\155\163\040\123\056\040\144\145\040\122\056\114\056\061\047\060
\045\006\003\125\004\013\014\036\124\162\165\163\164\103\157\162
\040\103\145\162\164\151\146\151\143\141\164\145\040\101\165\164
\150\157\162\151\164\171\061\037\060\035\006\003\125\004\003\014
\026\124\162\165\163\164\103\157\162\040\122\157\157\164\103\145
\162\164\040\103\101\055\062
END
CKA_SERIAL_NUMBER MULTILINE_OCTAL
\002\010\045\241\337\312\063\313\131\002
END
CKA_VALUE MULTILINE_OCTAL
\060\202\006\057\060\202\004\027\240\003\002\001\002\002\010\045
\241\337\312\063\313\131\002\060\015\006\011\052\206\110\206\367
\015\001\001\013\005\000\060\201\244\061\013\060\011\006\003\125
\004\006\023\002\120\101\061\017\060\015\006\003\125\004\010\014
\006\120\141\156\141\155\141\061\024\060\022\006\003\125\004\007
\014\013\120\141\156\141\155\141\040\103\151\164\171\061\044\060
\042\006\003\125\004\012\014\033\124\162\165\163\164\103\157\162
\040\123\171\163\164\145\155\163\040\123\056\040\144\145\040\122
\056\114\056\061\047\060\045\006\003\125\004\013\014\036\124\162
\165\163\164\103\157\162\040\103\145\162\164\151\146\151\143\141
\164\145\040\101\165\164\150\157\162\151\164\171\061\037\060\035
\006\003\125\004\003\014\026\124\162\165\163\164\103\157\162\040
\122\157\157\164\103\145\162\164\040\103\101\055\062\060\036\027
\015\061\066\060\062\060\064\061\062\063\062\062\063\132\027\015
\063\064\061\062\063\061\061\067\062\066\063\071\132\060\201\244
\061\013\060\011\006\003\125\004\006\023\002\120\101\061\017\060
\015\006\003\125\004\010\014\006\120\141\156\141\155\141\061\024
\060\022\006\003\125\004\007\014\013\120\141\156\141\155\141\040
\103\151\164\171\061\044\060\042\006\003\125\004\012\014\033\124
\162\165\163\164\103\157\162\040\123\171\163\164\145\155\163\040
\123\056\040\144\145\040\122\056\114\056\061\047\060\045\006\003
\125\004\013\014\036\124\162\165\163\164\103\157\162\040\103\145
\162\164\151\146\151\143\141\164\145\040\101\165\164\150\157\162
\151\164\171\061\037\060\035\006\003\125\004\003\014\026\124\162
\165\163\164\103\157\162\040\122\157\157\164\103\145\162\164\040
\103\101\055\062\060\202\002\042\060\015\006\011\052\206\110\206
\367\015\001\001\001\005\000\003\202\002\017\000\060\202\002\012
\002\202\002\001\000\247\040\156\302\052\242\142\044\225\220\166
\310\070\176\200\322\253\301\233\145\005\224\364\301\012\020\325
\002\254\355\237\223\307\207\310\260\047\053\102\014\075\012\076
\101\132\236\165\335\215\312\340\233\354\150\062\244\151\222\150
\214\013\201\016\126\240\076\032\335\054\045\024\202\057\227\323
\144\106\364\124\251\334\072\124\055\061\053\231\202\362\331\052
\327\357\161\000\270\061\244\276\172\044\007\303\102\040\362\212
\324\222\004\033\145\126\114\154\324\373\266\141\132\107\043\264
\330\151\264\267\072\320\164\074\014\165\241\214\116\166\241\351
\333\052\245\073\372\316\260\377\176\152\050\375\047\034\310\261
\351\051\361\127\156\144\264\320\301\025\155\016\276\056\016\106
\310\136\364\121\376\357\016\143\072\073\161\272\317\157\131\312
\014\343\233\135\111\270\114\342\127\261\230\212\102\127\234\166
\357\357\275\321\150\250\322\364\011\273\167\065\276\045\202\010
\304\026\054\104\040\126\251\104\021\167\357\135\264\035\252\136
\153\076\213\062\366\007\057\127\004\222\312\365\376\235\302\351
\350\263\216\114\113\002\061\331\344\074\110\202\047\367\030\202
\166\110\072\161\261\023\241\071\325\056\305\064\302\035\142\205
\337\003\376\115\364\257\075\337\134\133\215\372\160\341\245\176
\047\307\206\056\152\217\022\306\204\136\103\121\120\234\031\233
\170\346\374\366\355\107\176\173\075\146\357\023\023\210\137\074
\241\143\373\371\254\207\065\237\363\202\236\244\077\012\234\061
\151\213\231\244\210\112\216\156\146\115\357\026\304\017\171\050
\041\140\015\205\026\175\327\124\070\361\222\126\375\265\063\114
\203\334\327\020\237\113\375\306\370\102\275\272\174\163\002\340
\377\175\315\133\341\324\254\141\173\127\325\112\173\133\324\205
\130\047\135\277\370\053\140\254\240\046\256\024\041\047\306\167
\232\063\200\074\136\106\077\367\303\261\243\206\063\306\350\136
\015\271\065\054\252\106\301\205\002\165\200\240\353\044\373\025
\252\344\147\177\156\167\077\364\004\212\057\174\173\343\027\141
\360\335\011\251\040\310\276\011\244\320\176\104\303\262\060\112
\070\252\251\354\030\232\007\202\053\333\270\234\030\255\332\340
\106\027\254\317\135\002\003\001\000\001\243\143\060\141\060\035
\006\003\125\035\016\004\026\004\024\331\376\041\100\156\224\236
\274\233\075\234\175\230\040\031\345\214\060\142\262\060\037\006
\003\125\035\043\004\030\060\026\200\024\331\376\041\100\156\224
\236\274\233\075\234\175\230\040\031\345\214\060\142\262\060\017
\006\003\125\035\023\001\001\377\004\005\060\003\001\001\377\060
\016\006\003\125\035\017\001\001\377\004\004\003\002\001\206\060
\015\006\011\052\206\110\206\367\015\001\001\013\005\000\003\202
\002\001\000\236\105\236\014\073\266\357\341\072\310\174\321\000
\075\317\342\352\006\265\262\072\273\006\113\150\172\320\043\227
\164\247\054\360\010\330\171\132\327\132\204\212\330\022\232\033
\331\175\134\115\160\305\245\371\253\345\243\211\211\335\001\372
\354\335\371\351\222\227\333\260\106\102\363\323\142\252\225\376
\061\147\024\151\130\220\012\252\013\356\067\043\307\120\121\264
\365\176\236\343\173\367\344\314\102\062\055\111\014\313\377\111
\014\233\036\064\375\156\156\226\212\171\003\266\157\333\011\313
\375\137\145\024\067\341\070\365\363\141\026\130\344\265\155\015
\013\004\033\077\120\055\177\263\307\172\032\026\200\140\370\212
\037\351\033\052\306\371\272\001\032\151\277\322\130\307\124\127
\010\217\341\071\140\167\113\254\131\204\032\210\361\335\313\117
\170\327\347\341\063\055\374\356\101\372\040\260\276\313\367\070
\224\300\341\320\205\017\273\355\054\163\253\355\376\222\166\032
\144\177\133\015\063\011\007\063\173\006\077\021\244\134\160\074
\205\300\317\343\220\250\203\167\372\333\346\305\214\150\147\020
\147\245\122\055\360\304\231\217\177\277\321\153\342\265\107\326
\331\320\205\231\115\224\233\017\113\215\356\000\132\107\035\021
\003\254\101\030\257\207\267\157\014\072\217\312\317\334\003\301
\242\011\310\345\375\200\136\310\140\102\001\033\032\123\132\273
\067\246\267\274\272\204\351\036\154\032\324\144\332\324\103\376
\223\213\113\362\054\171\026\020\324\223\013\210\217\241\330\206
\024\106\221\107\233\050\044\357\127\122\116\134\102\234\252\367
\111\354\047\350\100\036\263\246\211\042\162\234\365\015\063\264
\130\243\060\073\335\324\152\124\223\276\032\115\363\223\224\367
\374\204\013\077\204\040\134\064\003\104\305\332\255\274\012\301
\002\317\036\345\224\331\363\216\133\330\114\360\235\354\141\027
\273\024\062\124\014\002\051\223\036\222\206\366\177\357\347\222
\005\016\131\335\231\010\056\056\372\234\000\122\323\305\146\051
\344\247\227\104\244\016\050\201\023\065\305\366\157\144\346\101
\304\325\057\314\064\105\045\317\101\000\226\075\112\056\302\226
\230\117\116\112\234\227\267\333\037\222\062\310\377\017\121\156
\326\354\011
END
CKA_NSS_MOZILLA_CA_POLICY CK_BBOOL CK_TRUE
# For Server Distrust After: Wed Nov 30 00:00:00 2022
CKA_NSS_SERVER_DISTRUST_AFTER MULTILINE_OCTAL
\xxxxxxxx
END
# For Email Distrust After: Wed Nov 30 00:00:00 2022
CKA_NSS_EMAIL_DISTRUST_AFTER MULTILINE_OCTAL
\062\062\061\061\063\060\060\060\060\060\060\060\132
END`
