/*---------------------------------------------------------*\
| CorsairHydroPlatinumController.cpp                        |
|                                                           |
|   Driver for Corsair Hydro Platinum coolers               |
|                                                           |
|   Kasper                                      28 Mar 2021 |
|   Nikola Jurkovic (jurkovic.nikola)           13 Aug 2025 |
|                                                           |
|   This file is part of the OpenRGB project                |
|   SPDX-License-Identifier: GPL-2.0-or-later               |
\*---------------------------------------------------------*/

#include <cstring>
#include "CorsairHydroPlatinumController.h"
#include "CorsairDeviceGuard.h"

static const uint8_t CRC_TABLE[256] =
{
    0x00, 0x07, 0x0E, 0x09, 0x1C, 0x1B, 0x12, 0x15,
    0x38, 0x3F, 0x36, 0x31, 0x24, 0x23, 0x2A, 0x2D,
    0x70, 0x77, 0x7E, 0x79, 0x6C, 0x6B, 0x62, 0x65,
    0x48, 0x4F, 0x46, 0x41, 0x54, 0x53, 0x5A, 0x5D,
    0xE0, 0xE7, 0xEE, 0xE9, 0xFC, 0xFB, 0xF2, 0xF5,
    0xD8, 0xDF, 0xD6, 0xD1, 0xC4, 0xC3, 0xCA, 0xCD,
    0x90, 0x97, 0x9E, 0x99, 0x8C, 0x8B, 0x82, 0x85,
    0xA8, 0xAF, 0xA6, 0xA1, 0xB4, 0xB3, 0xBA, 0xBD,
    0xC7, 0xC0, 0xC9, 0xCE, 0xDB, 0xDC, 0xD5, 0xD2,
    0xFF, 0xF8, 0xF1, 0xF6, 0xE3, 0xE4, 0xED, 0xEA,
    0xB7, 0xB0, 0xB9, 0xBE, 0xAB, 0xAC, 0xA5, 0xA2,
    0x8F, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9D, 0x9A,
    0x27, 0x20, 0x29, 0x2E, 0x3B, 0x3C, 0x35, 0x32,
    0x1F, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0D, 0x0A,
    0x57, 0x50, 0x59, 0x5E, 0x4B, 0x4C, 0x45, 0x42,
    0x6F, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7D, 0x7A,
    0x89, 0x8E, 0x87, 0x80, 0x95, 0x92, 0x9B, 0x9C,
    0xB1, 0xB6, 0xBF, 0xB8, 0xAD, 0xAA, 0xA3, 0xA4,
    0xF9, 0xFE, 0xF7, 0xF0, 0xE5, 0xE2, 0xEB, 0xEC,
    0xC1, 0xC6, 0xCF, 0xC8, 0xDD, 0xDA, 0xD3, 0xD4,
    0x69, 0x6E, 0x67, 0x60, 0x75, 0x72, 0x7B, 0x7C,
    0x51, 0x56, 0x5F, 0x58, 0x4D, 0x4A, 0x43, 0x44,
    0x19, 0x1E, 0x17, 0x10, 0x05, 0x02, 0x0B, 0x0C,
    0x21, 0x26, 0x2F, 0x28, 0x3D, 0x3A, 0x33, 0x34,
    0x4E, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5C, 0x5B,
    0x76, 0x71, 0x78, 0x7F, 0x6A, 0x6D, 0x64, 0x63,
    0x3E, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2C, 0x2B,
    0x06, 0x01, 0x08, 0x0F, 0x1A, 0x1D, 0x14, 0x13,
    0xAE, 0xA9, 0xA0, 0xA7, 0xB2, 0xB5, 0xBC, 0xBB,
    0x96, 0x91, 0x98, 0x9F, 0x8A, 0x8D, 0x84, 0x83,
    0xDE, 0xD9, 0xD0, 0xD7, 0xC2, 0xC5, 0xCC, 0xCB,
    0xE6, 0xE1, 0xE8, 0xEF, 0xFA, 0xFD, 0xF4, 0xF3
};

static const uint8_t MAGIC_1[61] =
{
    0x01, 0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F,
    0x7F, 0x7F, 0x7F, 0xFF, 0x00, 0xFF, 0xFF, 0xFF,
    0xFF, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0xFF,
    0xFF, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const uint8_t MAGIC_2[61] =
{
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const uint8_t MAGIC_3[61] =
{
    0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
    0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F,
    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
    0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const uint8_t HARDWARE_MAGIC_1[61] =
{
    0x00, 0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F,
    0x7F, 0x7F, 0x7F, 0x7F, 0x09, 0x20, 0x07, 0x00,
    0x0B, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0xFF,
    0xFF, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const uint8_t HARDWARE_MAGIC_2[61] =
{
    0x0A, 0x01, 0x04, 0x07, 0x05, 0x06, 0x07, 0x08,
    0x09, 0x0A, 0x0B, 0x00, 0x01, 0x02, 0x03, 0x04,
    0x01, 0x0A, 0x07, 0x04, 0x0B, 0x0A, 0x09, 0x08,
    0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00,
    0x01, 0x0A, 0x07, 0x04, 0x01, 0x0A, 0x09, 0x08,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const uint8_t HARDWARE_MAGIC_3[61] =
{
    0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00,
    0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
    0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
    0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
    0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const uint8_t HARDWARE_MAGIC_4[61] =
{
    0x00, 0x00, 0xFF, 0x00, 0x4A, 0xFF, 0x00, 0x94,
    0xFF, 0x00, 0xDF, 0xFF, 0x00, 0xFF, 0xAA, 0x00,
    0xFF, 0x15, 0x7F, 0x7F, 0x00, 0xFA, 0x00, 0x06,
    0xDB, 0x00, 0x32, 0xD7, 0x00, 0x58, 0xF6, 0x00,
    0x76, 0x94, 0x00, 0xB4, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00
};

CorsairHydroPlatinumController::CorsairHydroPlatinumController(hid_device* dev_handle, const char* path, bool dev_rgb_fan, std::string dev_name)
{
    dev                 = dev_handle;
    location            = path;
    name                = dev_name;
    have_rgb_fan        = dev_rgb_fan;
    guard_manager_ptr   = new DeviceGuardManager(new CorsairDeviceGuard());

    SendMagic(MAGIC_1, CORSAIR_HYDRO_PLATINUM_MAGIC_1);
    SendMagic(MAGIC_2, CORSAIR_HYDRO_PLATINUM_MAGIC_2);
    SendMagic(MAGIC_3, CORSAIR_HYDRO_PLATINUM_MAGIC_3);
}

CorsairHydroPlatinumController::~CorsairHydroPlatinumController()
{
    /*-----------------------------------------------------*\
    | Hardware lights, 2,3,4,1                              |
    \*-----------------------------------------------------*/
    SendMagic(HARDWARE_MAGIC_2, CORSAIR_HYDRO_PLATINUM_MAGIC_2);
    SendMagic(HARDWARE_MAGIC_3, CORSAIR_HYDRO_PLATINUM_MAGIC_3);
    SendMagic(HARDWARE_MAGIC_4, CORSAIR_HYDRO_PLATINUM_SET_LIGHTING_1);
    SendMagic(HARDWARE_MAGIC_1, CORSAIR_HYDRO_PLATINUM_MAGIC_1);

    hid_close(dev);
    delete guard_manager_ptr;
}

std::string CorsairHydroPlatinumController::GetLocation()
{
    return("HID: " + location);
}

std::string CorsairHydroPlatinumController::GetFirmwareString()
{
    return(firmware_version);
}

std::string CorsairHydroPlatinumController::GetName()
{
    return(name);
}

void CorsairHydroPlatinumController::SetupColors(std::vector<RGBColor> colors)
{
    unsigned int end_led = (colors.size() >= 20) ? 20 : (unsigned int)colors.size();
    SendColors(colors, 0, end_led, CORSAIR_HYDRO_PLATINUM_SET_LIGHTING_1);

    if(colors.size() > 20)
    {
        end_led = (colors.size() >= 40) ? 40 : (unsigned int)colors.size();
        SendColors(colors, 20, end_led, CORSAIR_HYDRO_PLATINUM_SET_LIGHTING_2);
    }
    if(colors.size() > 40)
    {
        end_led = (colors.size() >= 48) ? 48 : (unsigned int)colors.size();
        SendColors(colors, 40, end_led, CORSAIR_HYDRO_PLATINUM_SET_LIGHTING_3);
    }
}

bool CorsairHydroPlatinumController::HaveRgbFan()
{
    return(have_rgb_fan);
}

void CorsairHydroPlatinumController::SendMagic(const uint8_t* magic, unsigned int command)
{
    unsigned char usb_buf[CORSAIR_HYDRO_PLATINUM_PACKET_SIZE];

    /*-----------------------------------------------------*\
    | Set up config table request packet                    |
    \*-----------------------------------------------------*/
    usb_buf[0x00] = 0x00;
    usb_buf[0x01] = 0x3F;
    usb_buf[0x02] = (GetSequenceNumber()) | command;

    /*-----------------------------------------------------*\
    | Copy the magic bytes into the buffer                  |
    \*-----------------------------------------------------*/
    memcpy(&usb_buf[3], magic, 61 * sizeof magic[0]);

    /*-----------------------------------------------------*\
    | The data sent to the PEC function should not contain  |
    | the first (report id), second (prefix) and            |
    | last (checksum) bytes                                 |
    \*-----------------------------------------------------*/
    std::vector<unsigned char> checksum_array;
    checksum_array.insert(checksum_array.begin(), std::begin(usb_buf) + 2, std::end(usb_buf) - 1);
    usb_buf[64] = ComputePEC(static_cast<void*>(checksum_array.data()), 62);

    /*-----------------------------------------------------*\
    | Send packet                                           |
    \*-----------------------------------------------------*/
    /*---------------------------------------------------------*\
    | HID I/O start                                             |
    \*---------------------------------------------------------*/
    {
        DeviceGuardLock _ = guard_manager_ptr->AwaitExclusiveAccess();

        hid_write(dev, usb_buf, CORSAIR_HYDRO_PLATINUM_PACKET_SIZE);
        hid_read(dev, usb_buf, CORSAIR_HYDRO_PLATINUM_PACKET_SIZE);
    }
    /*---------------------------------------------------------*\
    | HID I/O end (lock released)                               |
    \*---------------------------------------------------------*/

    if(firmware_version.empty())
    {
        firmware_version =
            std::to_string(usb_buf[2] >> 4) + "."
            + std::to_string(usb_buf[2] & 0xf) + "."
            + std::to_string(usb_buf[3]);
    }

    /*-----------------------------------------------------*\
    | This delay prevents the AIO from soft-locking when    |
    | using an EE.                                          |
    \*-----------------------------------------------------*/
    std::this_thread::sleep_for(std::chrono::milliseconds(CORSAIR_HYDRO_PLATINUM_PACKET_DELAY));
}

void CorsairHydroPlatinumController::SendColors(std::vector<RGBColor> colors, unsigned int start, unsigned int end, unsigned int command)
{
    unsigned char usb_buf[CORSAIR_HYDRO_PLATINUM_PACKET_SIZE];

    /*-----------------------------------------------------*\
    | Zero out buffer                                       |
    \*-----------------------------------------------------*/
    memset(usb_buf, 0x00, CORSAIR_HYDRO_PLATINUM_PACKET_SIZE);

    /*-----------------------------------------------------*\
    | Set up config table request packet                    |
    \*-----------------------------------------------------*/
    usb_buf[0x00] = 0x00;
    usb_buf[0x01] = 0x3F;
    usb_buf[0x02] = (GetSequenceNumber()) | command;

    unsigned int i = 0;
    for(std::size_t color_idx = start; color_idx < end; color_idx++)
    {
        usb_buf[(i * 3) + 3] = RGBGetBValue(colors[color_idx]);
        usb_buf[(i * 3) + 4] = RGBGetGValue(colors[color_idx]);
        usb_buf[(i * 3) + 5] = RGBGetRValue(colors[color_idx]);
        i++;
    }

    /*-----------------------------------------------------*\
    | The data sent to the PEC function should not contain  |
    | the first (report id), second (prefix) and            |
    | last (checksum) bytes                                 |
    \*-----------------------------------------------------*/
    std::vector<unsigned char> checksum_array;
    checksum_array.insert(checksum_array.begin(), std::begin(usb_buf) + 2, std::end(usb_buf) - 1);
    usb_buf[64] = ComputePEC(static_cast<void*>(checksum_array.data()), 62);

    /*---------------------------------------------------------*\
    | HID I/O start                                             |
    \*---------------------------------------------------------*/
    {
        DeviceGuardLock _ = guard_manager_ptr->AwaitExclusiveAccess();

        hid_write(dev, usb_buf, CORSAIR_HYDRO_PLATINUM_PACKET_SIZE);
    }
    /*---------------------------------------------------------*\
    | HID I/O end (lock released)                               |
    \*---------------------------------------------------------*/

    /*-----------------------------------------------------*\
    | This delay prevents the AIO from soft-locking when    |
    | using an EE.                                          |
    \*-----------------------------------------------------*/
    std::this_thread::sleep_for(std::chrono::milliseconds(CORSAIR_HYDRO_PLATINUM_PACKET_DELAY));
}

unsigned int CorsairHydroPlatinumController::GetSequenceNumber()
{
    if(sequence_number < 31)
    {
        sequence_number++;
    }
    else
    {
        sequence_number = 1;
    }

    return(sequence_number << 3);
}

uint8_t CorsairHydroPlatinumController::ComputePEC(const void * data, size_t size)
{
	uint8_t val = 0;

	uint8_t * pos = (uint8_t *) data;
	uint8_t * end = pos + size;

	while(pos < end)
    {
		val = CRC_TABLE[val ^ *pos];
		pos++;
	}

	return val;
}
