/****************************************************************************/
// Eclipse SUMO, Simulation of Urban MObility; see https://eclipse.dev/sumo
// Copyright (C) 2012-2025 German Aerospace Center (DLR) and others.
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0/
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License 2.0 are satisfied: GNU General Public License, version 2
// or later which is available at
// https://www.gnu.org/licenses/old-licenses/gpl-2.0-standalone.html
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0-or-later
/****************************************************************************/
/// @file    TraCIDefs.h
/// @author  Daniel Krajzewicz
/// @author  Mario Krumnow
/// @author  Michael Behrisch
/// @author  Robert Hilbrich
/// @date    30.05.2012
///
// C++ TraCI client API implementation
/****************************************************************************/
#pragma once
// we do not include config.h here, since we should be independent of a special sumo build
// but we want to avoid certain warnings in MSVC see config.h.cmake for details
#ifdef _MSC_VER
#pragma warning(push)
#pragma warning(disable: 4514 4820)
#endif

#include <libsumo/TraCIConstants.h>
#include <vector>
#include <limits>
#include <map>
#include <string>
#include <stdexcept>
#include <sstream>
#include <memory>
#include <cstring>


// ===========================================================================
// common declarations
// ===========================================================================
namespace libsumo {
class VariableWrapper;
}
namespace tcpip {
class Storage;
}


// ===========================================================================
// global definitions
// ===========================================================================
#ifdef LIBTRACI
#define LIBSUMO_NAMESPACE libtraci
#else
#define LIBSUMO_NAMESPACE libsumo
#endif

#define LIBSUMO_SUBSCRIPTION_API \
static void subscribe(const std::string& objectID, const std::vector<int>& varIDs = std::vector<int>({-1}), \
                      double begin = libsumo::INVALID_DOUBLE_VALUE, double end = libsumo::INVALID_DOUBLE_VALUE, const libsumo::TraCIResults& parameters = libsumo::TraCIResults()); \
static void unsubscribe(const std::string& objectID); \
static void subscribeContext(const std::string& objectID, int domain, double dist, const std::vector<int>& varIDs = std::vector<int>({-1}), \
                             double begin = libsumo::INVALID_DOUBLE_VALUE, double end = libsumo::INVALID_DOUBLE_VALUE, const libsumo::TraCIResults& parameters = libsumo::TraCIResults()); \
static void unsubscribeContext(const std::string& objectID, int domain, double dist); \
static const libsumo::SubscriptionResults getAllSubscriptionResults(); \
static const libsumo::TraCIResults getSubscriptionResults(const std::string& objectID); \
static const libsumo::ContextSubscriptionResults getAllContextSubscriptionResults(); \
static const libsumo::SubscriptionResults getContextSubscriptionResults(const std::string& objectID); \
static void subscribeParameterWithKey(const std::string& objectID, const std::string& key, double beginTime = libsumo::INVALID_DOUBLE_VALUE, double endTime = libsumo::INVALID_DOUBLE_VALUE); \
static const int DOMAIN_ID; \
static int domainID() { return DOMAIN_ID; }

#define LIBSUMO_SUBSCRIPTION_IMPLEMENTATION(CLASS, DOM) \
const int CLASS::DOMAIN_ID(libsumo::CMD_GET_##DOM##_VARIABLE); \
void \
CLASS::subscribe(const std::string& objectID, const std::vector<int>& varIDs, double begin, double end, const libsumo::TraCIResults& parameters) { \
    libsumo::Helper::subscribe(libsumo::CMD_SUBSCRIBE_##DOM##_VARIABLE, objectID, varIDs, begin, end, parameters); \
} \
void \
CLASS::unsubscribe(const std::string& objectID) { \
    libsumo::Helper::subscribe(libsumo::CMD_SUBSCRIBE_##DOM##_VARIABLE, objectID, std::vector<int>(), libsumo::INVALID_DOUBLE_VALUE, libsumo::INVALID_DOUBLE_VALUE, libsumo::TraCIResults()); \
} \
void \
CLASS::subscribeContext(const std::string& objectID, int domain, double dist, const std::vector<int>& varIDs, double begin, double end, const TraCIResults& parameters) { \
    libsumo::Helper::subscribe(libsumo::CMD_SUBSCRIBE_##DOM##_CONTEXT, objectID, varIDs, begin, end, parameters, domain, dist); \
} \
void \
CLASS::unsubscribeContext(const std::string& objectID, int domain, double dist) { \
    libsumo::Helper::subscribe(libsumo::CMD_SUBSCRIBE_##DOM##_CONTEXT, objectID, std::vector<int>(), libsumo::INVALID_DOUBLE_VALUE, libsumo::INVALID_DOUBLE_VALUE, libsumo::TraCIResults(), domain, dist); \
} \
const libsumo::SubscriptionResults \
CLASS::getAllSubscriptionResults() { \
    return mySubscriptionResults; \
} \
const libsumo::TraCIResults \
CLASS::getSubscriptionResults(const std::string& objectID) { \
    return mySubscriptionResults[objectID]; \
} \
const libsumo::ContextSubscriptionResults \
CLASS::getAllContextSubscriptionResults() { \
    return myContextSubscriptionResults; \
} \
const libsumo::SubscriptionResults \
CLASS::getContextSubscriptionResults(const std::string& objectID) { \
    return myContextSubscriptionResults[objectID]; \
} \
void \
CLASS::subscribeParameterWithKey(const std::string& objectID, const std::string& key, double beginTime, double endTime) { \
    libsumo::Helper::subscribe(libsumo::CMD_SUBSCRIBE_##DOM##_VARIABLE, objectID, std::vector<int>({libsumo::VAR_PARAMETER_WITH_KEY}), beginTime, endTime, libsumo::TraCIResults {{libsumo::VAR_PARAMETER_WITH_KEY, std::make_shared<libsumo::TraCIString>(key)}}); \
}


#define LIBSUMO_ID_PARAMETER_API \
static std::vector<std::string> getIDList(); \
static int getIDCount(); \
static std::string getParameter(const std::string& objectID, const std::string& key); \
static const std::pair<std::string, std::string> getParameterWithKey(const std::string& objectID, const std::string& key); \
static void setParameter(const std::string& objectID, const std::string& key, const std::string& value);

#define LIBSUMO_GET_PARAMETER_WITH_KEY_IMPLEMENTATION(CLASS) \
const std::pair<std::string, std::string> \
CLASS::getParameterWithKey(const std::string& objectID, const std::string& key) { \
    return std::make_pair(key, getParameter(objectID, key)); \
}


#define SWIGJAVA_CAST(CLASS) \
static std::shared_ptr<CLASS> cast(std::shared_ptr<TraCIResult> res) { \
    return std::dynamic_pointer_cast<CLASS>(res); \
}


// ===========================================================================
// class and type definitions
// ===========================================================================
namespace libsumo {
/**
 * @class TraCIException
 * @brief An error which allows to continue
 */
class TraCIException : public std::runtime_error {
public:
    /** constructor */
    TraCIException(std::string what)
        : std::runtime_error(what) {}
};

/**
 * @class FatalTraCIError
 * @brief An error which is not recoverable
 */
class FatalTraCIError : public std::runtime_error {
public:
    /** constructor */
    FatalTraCIError(std::string what)
        : std::runtime_error(what) {}
};

/// @name Structures definitions
/// @{

struct TraCIResult {
    virtual ~TraCIResult() {}
    virtual std::string getString() const {
        return "";
    }
    virtual int getType() const {
        return -1;
    }
};

/** @struct TraCIPosition
 * @brief A 2D or 3D-position, for 2D positions z == INVALID_DOUBLE_VALUE
 */
struct TraCIPosition : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCIPosition(" << x << "," << y;
        if (z != INVALID_DOUBLE_VALUE) {
            os << "," << z;
        }
        os << ")";
        return os.str();
    }
    int getType() const override {
        return z != INVALID_DOUBLE_VALUE ? POSITION_3D : POSITION_2D;
    }
    double x = INVALID_DOUBLE_VALUE, y = INVALID_DOUBLE_VALUE, z = INVALID_DOUBLE_VALUE;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIPosition)
#endif
};

/** @struct TraCIRoadPosition
 * @brief An edgeId, position and laneIndex
 */
struct TraCIRoadPosition : TraCIResult {
    TraCIRoadPosition(const std::string e = "", const double p = INVALID_DOUBLE_VALUE, const int li = INVALID_INT_VALUE) : edgeID(e), pos(p), laneIndex(li) {}
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCIRoadPosition(" << edgeID << "_" << laneIndex << "," << pos << ")";
        return os.str();
    }
    int getType() const override {
        return POSITION_ROADMAP;
    }
    std::string edgeID;
    double pos;
    int laneIndex;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIRoadPosition)
#endif
};

/** @struct TraCIColor
 * @brief A color
 */
struct TraCIColor : TraCIResult {
    TraCIColor() : r(0), g(0), b(0), a(255) {}
    TraCIColor(int r, int g, int b, int a = 255) : r(r), g(g), b(b), a(a) {}
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCIColor(" << r << "," << g << "," << b << "," << a << ")";
        return os.str();
    }
    int r, g, b, a;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIColor)
#endif
};


/** @struct TraCIPositionVector
 * @brief A list of positions
 */
struct TraCIPositionVector : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "[";
        for (const TraCIPosition& v : value) {
            os << "(" << v.x << "," << v.y << "," << v.z << ")";
        }
        os << "]";
        return os.str();
    }
    std::vector<TraCIPosition> value;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIPositionVector)
#endif
};


struct TraCIInt : TraCIResult {
    TraCIInt(int v = 0, int t = libsumo::TYPE_INTEGER) : value(v), traciType(t) {}
    std::string getString() const override {
        std::ostringstream os;
        os << value;
        return os.str();
    }
    int getType() const override {
        return traciType;
    }
    int value;
    int traciType;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIInt)
#endif
};


struct TraCIDouble : TraCIResult {
    TraCIDouble(double v = 0.) : value(v) {}
    std::string getString() const override {
        std::ostringstream os;
        os << value;
        return os.str();
    }
    int getType() const override {
        return libsumo::TYPE_DOUBLE;
    }
    double value;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIDouble)
#endif
};


struct TraCIString : TraCIResult {
    TraCIString(std::string v = "") : value(v) {}
    std::string getString() const override {
        return value;
    }
    int getType() const override {
        return libsumo::TYPE_STRING;
    }
    std::string value;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIString)
#endif
};


struct TraCIStringList : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "[";
        for (std::string v : value) {
            os << v << ",";
        }
        os << "]";
        return os.str();
    }
    std::vector<std::string> value;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIStringList)
#endif
};


struct TraCIDoubleList : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "[";
        for (double v : value) {
            os << v << ",";
        }
        os << "]";
        return os.str();
    }
    std::vector<double> value;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIDoubleList)
#endif
};


struct TraCIIntList : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "[";
        for (int v : value) {
            os << v << ",";
        }
        os << "]";
        return os.str();
    }
    std::vector<int> value;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIIntList)
#endif
};


struct TraCIStringDoublePairList : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "[";
        for (const auto& v : value) {
            os << "(" << v.first << "," << v.second << "),";
        }
        os << "]";
        return os.str();
    }
    std::vector<std::pair<std::string, double> > value;
#ifdef SWIGJAVA
    SWIGJAVA_CAST(TraCIStringDoublePairList)
#endif
};


/// @brief {variable->value}
typedef std::map<int, std::shared_ptr<libsumo::TraCIResult> > TraCIResults;
/// @brief {object->{variable->value}}
typedef std::map<std::string, libsumo::TraCIResults> SubscriptionResults;
typedef std::map<std::string, libsumo::SubscriptionResults> ContextSubscriptionResults;


struct TraCIPhase {
    TraCIPhase() {}
    TraCIPhase(const double _duration, const std::string& _state, const double _minDur = libsumo::INVALID_DOUBLE_VALUE,
               const double _maxDur = libsumo::INVALID_DOUBLE_VALUE,
               const std::vector<int>& _next = std::vector<int>(),
               const std::string& _name = "",
               const std::string& _earlyTarget = "") :
        duration(_duration), state(_state), minDur(_minDur), maxDur(_maxDur), next(_next), name(_name), earlyTarget(_earlyTarget) {}
    ~TraCIPhase() {}

    double duration;
    std::string state;
    double minDur, maxDur;
    std::vector<int> next;
    std::string name;
    std::string earlyTarget;
};
}


#ifdef SWIG
%template(TraCIPhaseVector) std::vector<std::shared_ptr<libsumo::TraCIPhase> >; // *NOPAD*
#endif


namespace libsumo {
struct TraCILogic {
    TraCILogic() {}
    TraCILogic(const std::string& _programID, const int _type, const int _currentPhaseIndex,
               const std::vector<std::shared_ptr<libsumo::TraCIPhase> >& _phases = std::vector<std::shared_ptr<libsumo::TraCIPhase> >())
        : programID(_programID), type(_type), currentPhaseIndex(_currentPhaseIndex), phases(_phases) {}
    ~TraCILogic() {}

    std::string getString() const {
        std::ostringstream os;
        os << "TraCILink(" << programID << "," << type << "," << currentPhaseIndex << ")";
        return os.str();
    }

    std::string programID;
    int type;
    int currentPhaseIndex;
    std::vector<std::shared_ptr<libsumo::TraCIPhase> > phases;
    std::map<std::string, std::string> subParameter;
};


struct TraCILogicVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCILogicVectorWrapped[";
        for (const TraCILogic& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCILogic> value;
};


struct TraCILink {
    TraCILink() {}
    TraCILink(const std::string& _from, const std::string& _via, const std::string& _to)
        : fromLane(_from), viaLane(_via), toLane(_to) {}
    ~TraCILink() {}

    std::string getString() const {
        std::ostringstream os;
        os << "TraCILink(" << fromLane << "," << viaLane << "," << toLane << ")";
        return os.str();
    }

    std::string fromLane;
    std::string viaLane;
    std::string toLane;
};


struct TraCILinkVectorVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCILinkVectorVectorWrapped[";
        for (const std::vector<TraCILink>& v : value) {
            os << "[";
            for (const TraCILink& tl : v) {
                os << tl.getString() << ",";
            }
        }
        os << "]";
        return os.str();
    }

    std::vector<std::vector<TraCILink> > value;
};


struct TraCIConnection {
    TraCIConnection() {} // this is needed by SWIG when building a vector of this type, please don't use it
    TraCIConnection(const std::string& _approachedLane, const bool _hasPrio, const bool _isOpen, const bool _hasFoe,
                    const std::string _approachedInternal, const std::string _state, const std::string _direction, const double _length)
        : approachedLane(_approachedLane), hasPrio(_hasPrio), isOpen(_isOpen), hasFoe(_hasFoe),
          approachedInternal(_approachedInternal), state(_state), direction(_direction), length(_length) {}
    ~TraCIConnection() {}

    std::string getString() const {
        std::ostringstream os;
        os << "TraCIConnection(" << approachedLane << "," << hasPrio << "," << isOpen
           << "," << hasFoe << "," << approachedInternal << "," << state << "," << direction << "," << length << ")";
        return os.str();
    }

    std::string approachedLane;
    bool hasPrio;
    bool isOpen;
    bool hasFoe;
    std::string approachedInternal;
    std::string state;
    std::string direction;
    double length;
};


struct TraCIConnectionVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCIConnectionVectorWrapped[";
        for (const TraCIConnection& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCIConnection> value;
};


/// @brief mirrors MSInductLoop::VehicleData
struct TraCIVehicleData {
    std::string getString() const {
        std::ostringstream os;
        os << "TraCIVehicleData(" << id << "," << length << "," << entryTime
           << "," << leaveTime << "," << typeID << ")";
        return os.str();
    }

    /// @brief The id of the vehicle
    std::string id;
    /// @brief Length of the vehicle
    double length;
    /// @brief Entry-time of the vehicle in [s]
    double entryTime;
    /// @brief Leave-time of the vehicle in [s]
    double leaveTime;
    /// @brief Type of the vehicle in
    std::string typeID;
};


struct TraCIVehicleDataVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCIVehicleDataVectorWrapped[";
        for (const TraCIVehicleData& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCIVehicleData> value;
};


struct TraCINextTLSData {
    std::string getString() const {
        std::ostringstream os;
        os << "TraCINextTLSData(" << id << "," << tlIndex << "," << dist
           << "," << state << ")";
        return os.str();
    }

    /// @brief The id of the next tls
    std::string id;
    /// @brief The tls index of the controlled link
    int tlIndex;
    /// @brief The distance to the tls
    double dist;
    /// @brief The current state of the tls
    char state;
};


struct TraCINextTLSDataVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCINextTLSDataVectorWrapped[";
        for (const TraCINextTLSData& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCINextTLSData> value;
};


struct TraCINextStopData {

    TraCINextStopData(const std::string& lane = "",
                      double startPos = INVALID_DOUBLE_VALUE,
                      double endPos = INVALID_DOUBLE_VALUE,
                      const std::string& stoppingPlaceID = "",
                      int stopFlags = 0,
                      double duration = INVALID_DOUBLE_VALUE,
                      double until = INVALID_DOUBLE_VALUE,
                      double intendedArrival = INVALID_DOUBLE_VALUE,
                      double arrival = INVALID_DOUBLE_VALUE,
                      double depart = INVALID_DOUBLE_VALUE,
                      const std::string& split = "",
                      const std::string& join = "",
                      const std::string& actType = "",
                      const std::string& tripId = "",
                      const std::string& line = "",
                      double speed = 0):
        lane(lane),
        startPos(startPos),
        endPos(endPos),
        stoppingPlaceID(stoppingPlaceID),
        stopFlags(stopFlags),
        duration(duration),
        until(until),
        intendedArrival(intendedArrival),
        arrival(arrival),
        depart(depart),
        split(split),
        join(join),
        actType(actType),
        tripId(tripId),
        line(line),
        speed(speed)
    {}

    std::string getString() const {
        std::ostringstream os;
        os << "TraCINextStopData(" << lane << "," << endPos << "," << stoppingPlaceID
           << "," << stopFlags << "," << duration << "," << until
           << "," << arrival << ")";
        return os.str();
    }

    /// @brief The lane to stop at
    std::string lane;
    /// @brief The stopping position start
    double startPos;
    /// @brief The stopping position end
    double endPos;
    /// @brief Id assigned to the stop
    std::string stoppingPlaceID;
    /// @brief Stop flags
    int stopFlags;
    /// @brief The intended (minimum) stopping duration
    double duration;
    /// @brief The time at which the vehicle may continue its journey
    double until;
    /// @brief The intended arrival time
    double intendedArrival;
    /// @brief The actual arrival time (only for past stops)
    double arrival;
    /// @brief The time at which this stop was ended
    double depart;
    /// @brief the id of the vehicle (train portion) that splits of upon reaching this stop
    std::string split;
    /// @brief the id of the vehicle (train portion) to which this vehicle shall be joined
    std::string join;
    /// @brief additional information for this stop
    std::string actType;
    /// @brief id of the trip within a cyclical public transport route
    std::string tripId;
    /// @brief the new line id of the trip within a cyclical public transport route
    std::string line;
    /// @brief the speed at which this stop counts as reached (waypoint mode)
    double speed;
};


/** @struct TraCINextStopDataVectorWrapped
 * @brief A list of vehicle stops
 * @see TraCINextStopData
 */
struct TraCINextStopDataVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCINextStopDataVectorWrapped[";
        for (const TraCINextStopData& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCINextStopData> value;
};


struct TraCIBestLanesData {
    std::string getString() const {
        std::ostringstream os;
        os << "TraCIBestLanesData(" << laneID << "," << length << "," << occupation
           << "," << bestLaneOffset << "," << allowsContinuation << ",[";
        for (const std::string& s : continuationLanes) {
            os << s << ",";
        }
        os << "])";
        return os.str();
    }

    /// @brief The id of the lane
    std::string laneID;
    /// @brief The length than can be driven from that lane without lane change
    double length;
    /// @brief The traffic density along length
    double occupation;
    /// @brief The offset of this lane from the best lane
    int bestLaneOffset;
    /// @brief Whether this lane allows continuing the route
    bool allowsContinuation;
    /// @brief The sequence of lanes that best allows continuing the route without lane change
    std::vector<std::string> continuationLanes;
};


struct TraCIBestLanesDataVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCIBestLanesDataVectorWrapped[";
        for (const TraCIBestLanesData& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCIBestLanesData> value;
};


struct TraCIStage : TraCIResult {
public:
    TraCIStage(int type = INVALID_INT_VALUE, const std::string& vType = "", const std::string& line = "", const std::string& destStop = "",
               const std::vector<std::string>& edges = std::vector<std::string>(),
               double travelTime = INVALID_DOUBLE_VALUE, double cost = INVALID_DOUBLE_VALUE, double length = INVALID_DOUBLE_VALUE,
               const std::string& intended = "", double depart = INVALID_DOUBLE_VALUE, double departPos = INVALID_DOUBLE_VALUE,
               double arrivalPos = INVALID_DOUBLE_VALUE, const std::string& description = "") :
        type(type), vType(vType), line(line), destStop(destStop), edges(edges), travelTime(travelTime), cost(cost),
        length(length), intended(intended), depart(depart), departPos(departPos), arrivalPos(arrivalPos), description(description) {}
    /// @brief The type of stage (walking, driving, ...)
    int type;
    /// @brief The vehicle type when using a private car or bike
    std::string vType;
    /// @brief The line or the id of the vehicle type
    std::string line;
    /// @brief The id of the destination stop
    std::string destStop;
    /// @brief The sequence of edges to travel
    std::vector<std::string> edges;
    /// @brief duration of the stage in seconds
    double travelTime;
    /// @brief effort needed
    double cost;
    /// @brief length in m
    double length;
    /// @brief id of the intended vehicle for public transport ride
    std::string intended;
    /// @brief intended depart time for public transport ride or INVALID_DOUBLE_VALUE
    double depart;
    /// @brief position on the lane when starting the stage
    double departPos;
    /// @brief position on the lane when ending the stage
    double arrivalPos;
    /// @brief arbitrary description string
    std::string description;
};



struct TraCIReservation {
    TraCIReservation() {}
    TraCIReservation(const std::string& id,
                     const std::vector<std::string>& persons,
                     const std::string& group,
                     const std::string& fromEdge,
                     const std::string& toEdge,
                     double departPos,
                     double arrivalPos,
                     double depart,
                     double reservationTime,
                     int state) :
        id(id), persons(persons), group(group), fromEdge(fromEdge), toEdge(toEdge), departPos(departPos), arrivalPos(arrivalPos),
        depart(depart), reservationTime(reservationTime), state(state) {}
    /// @brief The id of the taxi reservation (usable for traci.vehicle.dispatchTaxi)
    std::string id;
    /// @brief The persons ids that are part of this reservation
    std::vector<std::string> persons;
    /// @brief The group id of this reservation
    std::string group;
    /// @brief The origin edge id
    std::string fromEdge;
    /// @brief The destination edge id
    std::string toEdge;
    /// @brief pickup position on the origin edge
    double departPos;
    /// @brief drop-off position on the destination edge
    double arrivalPos;
    /// @brief pickup-time
    double depart;
    /// @brief time when the reservation was made
    double reservationTime;
    /// @brief the state of this reservation
    int state;

    std::string getString() const {
        std::ostringstream os;
        os << "TraCIReservation(id=" << id << ")";
        return os.str();
    }
};


struct TraCIReservationVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCIReservationVectorWrapped[";
        for (const TraCIReservation& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCIReservation> value;
};


struct TraCICollision {
    /// @brief The ids of the participating vehicles and persons
    std::string collider;
    std::string victim;
    std::string colliderType;
    std::string victimType;
    double colliderSpeed;
    double victimSpeed;
    /// @brief The type of collision
    std::string type;
    /// @brief The lane where the collision happended
    std::string lane;
    /// @brief The position of the collision along the lane
    double pos;

    std::string getString() const {
        std::ostringstream os;
        os << "TraCICollision(collider=" << collider << ", victim=" << victim << ")";
        return os.str();
    }
};


struct TraCICollisionVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCICollisionVectorWrapped[";
        for (const TraCICollision& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCICollision> value;
};


struct TraCISignalConstraint {
    /// @brief the idea of the rail signal where this constraint is active
    std::string signalId;
    /// @brief the tripId or vehicle id of the train that is constrained
    std::string tripId;
    /// @brief the tripId or vehicle id of the train that must pass first
    std::string foeId;
    /// @brief the tlsID of the rail signla that the foe must pass first
    std::string foeSignal;
    /// @brief the number of trains that must be recorded at the foeSignal
    int limit;
    /// @brief the type of constraint (predecessor:0, insertionPredecessor:1)
    int type;
    /// @brief whether tripId must still wait for foeId to pass foeSignal
    bool mustWait;
    /// @brief whether this constraint is active
    bool active;
    /// @brief additional parameters
    std::map<std::string, std::string> param;

    std::string getString() const {
        std::ostringstream os;
        os << "TraCISignalConstraint(signalId=" << signalId << ", tripid=" << tripId << ", foeSignal=" << foeSignal << ", foeId=" << foeId << ")";
        return os.str();
    }
};


struct TraCISignalConstraintVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCISignalConstraintVectorWrapped[";
        for (const TraCISignalConstraint& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCISignalConstraint> value;
};


struct TraCIJunctionFoe {
    /// @brief the id of the vehicle with intersecting trajectory
    std::string foeId;
    double egoDist;
    double foeDist;
    double egoExitDist;
    double foeExitDist;
    std::string egoLane;
    std::string foeLane;
    bool egoResponse;
    bool foeResponse;

    std::string getString() const {
        std::ostringstream os;
        os << "TraCIJunctionFoe(foeId=" << foeId << ", egoDist=" << egoDist << ", foeDist=" << foeDist << ", foeDist=" << foeDist << ")";
        return os.str();
    }
};


struct TraCIJunctionFoeVectorWrapped : TraCIResult {
    std::string getString() const override {
        std::ostringstream os;
        os << "TraCIJunctionFoeVectorWrapped[";
        for (const TraCIJunctionFoe& v : value) {
            os << v.getString() << ",";
        }
        os << "]";
        return os.str();
    }

    std::vector<TraCIJunctionFoe> value;
};


}

// pop MSVC warnings
#ifdef _MSC_VER
#pragma warning(pop)
#endif
